//===========================================================================
//
// Copyright (c)  2002-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.xml.decryptionTransform;

import java.security.PrivateKey;

import org.w3c.dom.Document;

import iaik.ixsil.algorithms.CanonicalizationAlgorithmImplCanonicalXML;
import iaik.ixsil.algorithms.DigestAlgorithmImplSHA1;
import iaik.ixsil.algorithms.TransformImplEnvelopedSignature;
import iaik.ixsil.algorithms.TransformImplXPath;
import iaik.ixsil.core.Position;
import iaik.ixsil.core.Signer;
import iaik.ixsil.core.SignerReference;
import iaik.ixsil.core.SignerSignature;
import iaik.ixsil.core.SignerSignedInfo;
import iaik.ixsil.exceptions.ReferenceException;
import iaik.ixsil.exceptions.SignedInfoException;
import iaik.ixsil.exceptions.URIException;
import iaik.ixsil.init.IXSILInit;
import iaik.ixsil.util.ExternalReferenceResolverImpl;
import iaik.ixsil.util.URI;
import iaik.x509.X509Certificate;

import com.entrust.toolkit.KeyAndCertificateSource;
import com.entrust.toolkit.User;
import com.entrust.toolkit.examples.xml.utils.Utils;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.xencrypt.core.TransformImplDecryption;

/**
 * This class demonstrates the use of the Decryption Transform when signing XML.
 * The Decryption Transform allows signed content to be encrypted after it has been
 * signed;  the Toolkit's <CODE>Verifier</CODE> uses the Decryption Transform to
 * determine which portions of the signed content were already encrypted before
 * they were signed.  Internally, the Toolkit decrypts the other encryption portions
 * as it verifies the signature.
 * <P>
 * The Toolkit's implementation of the Decryption Transform follows the specification in
 * <a href="http://www.w3.org/TR/2001/WD-xmlenc-decrypt-20011018">Decryption Transform
 * for XML Signature, W3C Working Draft 18 October 2001</a>.
 * </P>
 * <P>
 * This sample program creates an Enveloped XML signature with an XPath transform and
 * a Decryption Transform.  The XPath transform enables this sample to sign only a portion
 * of the document.  As an example, we sign a portion that contains some encrypted elements,
 * while leaving other encrypted elements outside the signed content.  After signing, we
 * encrypt additional elements within the signed content.  This demonstrates how the Toolkit's
 * <CODE>Verifier</CODE> decrypts only the content it needs to verify the signature.
 * </P>
 *@see VerifyDecryptionTransform
 */
public class SignDecryptionTransform
{
  /**
   * A source of keys and certificates
   */
  private KeyAndCertificateSource m_source ;
  
  /**
   * The main method.
   */
  public static void main(String[] args)
  throws Exception
  {
    SignDecryptionTransform signatureExample = new SignDecryptionTransform(args);

    // Optional -- display the XML resource to be signed.
    Utils.displayURL(args[3]);

    // Find where to insert the Signature
    String insertSignatureHere = args.length > 5 ? args[5] : null ;

    // Read the optional command line argument that specifies which nodes to sign.
    // If none was provided, sign all the nodes.
    // e.g. "true()" would sign all elements (same effect as omitting the XPath transform)
    // e.g. "ancestor-or-self::SignThis" would sign elements named <SignThis>;
    String xPath = args.length > 6 ? args[6] : null ;

    // Sign the document.
    Document signatureDOMDoc = signatureExample.createSignature(args[3],
                                                                insertSignatureHere,
                                                                xPath);
    // Write the signed document to a disk file
    Utils.writeDocumentToFile(args[4], signatureDOMDoc);

    // Optional -- Display signed document.
    Utils.displayDocument(signatureDOMDoc, args[4]) ;

    System.out.println("Signed.");
  }

  /**
   * Constructor for this example. Logs into a User object, so a private signing key
   * is accessible, and initializes the IXSIL library.
   */
  public SignDecryptionTransform(String[] args)
  throws Exception
  {
    if (args.length < 5 || args.length > 7)
    {
      System.out.println("\nPlease specify the following command line parameters, number (6) being optional:");
      System.out.println("  (1) Path and filename of an Entrust user profile");
      System.out.println("  (2) Profile password");
      System.out.println("  (3) The URL of an IXSIL initialization properties file\n\t" +
                                "e.g. \"file:/C:/etjava/examples/xml/init/properties/init.properties\"\n\t" +
                                "e.g. \"init.properties\"\n\t" +
                                "e.g. \"http://hostname/init.properties\"");
      System.out.println("  (4) URI of an XML resource to be signed, e.g. \"file:/c:/test/tobesigned.xml\"");
      System.out.println("  (5) Path and filename for the resulting XML signature file");
      System.out.println("  (6) [Name of an XML element in (4) where the Signature is to be inserted]");
      System.out.println("  (7) [An XPath expression that will select the nodes to be signed]");
      System.exit(0);
    }

    // Need the signing key and certificate to be in a KeyAndCertificateSource wrapper.
    m_source = new KeyAndCertificateSource();
    
    // If you using a keystore such as P12, PFX, Entrust EPF, or java.security.KeyStore,
    // then log in here, to get access to its signing key and certificate.
    User user = Utils.login(args[0], new SecureStringBuffer(new StringBuffer(args[1])));
    PrivateKey pk = user.getSigningKey();
    X509Certificate cert = user.getVerificationCertificate();
    
    // In any case, provide a signing key and verification certificate from whatever source you use.
    m_source.setSigningInfo(pk, cert);

    // The properties file is specified by a URI.  You can provide the name of the
    // file, e.g 'init.properties', to the URI constructor.  In that case, the folder
    // that contains init.properties must be in the classpath.
    // Please refer to the comments in the 'init.properties' sample.
    URI initProps = new URI(args[2]);

    // Initialize IXSIL
    System.out.println("Initializing IXSIL properties from \"" + initProps + "\"...");
    IXSILInit.init(initProps);
  }

  /**
   * Creates an Enveloped Signature with a Decryption Transform and an XPath Transform.
   *
   * @param uriTobeSigned The URI of an XML resource to be signed, e.g "file:///c:/test/tobesigned.xml"
   *
   * @param insertSignatureHere The name of an XML element in the resource to be signed.  The Signature
   *                            will be inserted at that position.  If null, the signature will be inserted
   *                            as the first child element under the document root.
   *
   * @param xpathExpr An XPath that specifies which nodes to sign.  If null, the entire document will be signed.
   *
   * @return A DOM document that is the same as the resource to be signed but with a Signature element inserted.
   */
  public Document createSignature(String uriTobeSigned, String insertSignatureHere, String xpathExpr)
  throws Exception
  {
    // Define a position in the XML document where you want to insert the Signature.
    // The position can be arbitrary.  If the user did not specify an element by name
    // on the command line, insert the signature as the first child of the document root.
    String xpathSignatureElement = null ;
    if(insertSignatureHere != null) {
        xpathSignatureElement = "//" + insertSignatureHere + "[1]";
    }
    else {
        xpathSignatureElement = "/*[1]";
    }
    System.out.println("Inserting dsig:Signature element at:" + xpathSignatureElement);

    Position signaturePosition = new Position(xpathSignatureElement, "", 0) ;
    URI baseURI = new URI(uriTobeSigned);

    // Create a Signer, using the constructor that inserts a Signature
    // element into an existing document.
    Signer signer = new Signer(
        (new ExternalReferenceResolverImpl(baseURI)).resolve(baseURI),
        baseURI,
        signaturePosition);

    // Optional -- show where we will insert the Signature element.
    // The SignedInfo is empty so far.
    // displayDocument(signer.toDocument(), "");

    // Get an interface for the signature
    SignerSignature signature = signer.getSignature();

    // Get an interface for signed information
    SignerSignedInfo signedInfo = signer.getSignature().getSignerSignedInfo();

    // Give the Signature element an ID, so the verifier can locate it.
    signature.setId("Signature001");

    // Set the canonicalization algorithm
    setCanonicalizationAlgorithm(signedInfo);

    // Configure the signature algorithm (RSA or DSA)
    Utils.configureAlgorithm(m_source, signedInfo);

    // Specify what content will be signed and the digest algorithm to use
    configureReference(signedInfo, xpathExpr);

    // Attach the signer's certificate to the <Signature>
    Utils.setX509KeyInfo(Utils.X509DATA_CERTIFICATE, signer, m_source.getVerificationCertificate(), null);

    // Compute the signature value
    signer.getSignature().sign();

    // Optional, display the document
    // displayDocument(signer.toDocument(), "");

    // Return the signed document
    return signer.toDocument();
  }

  /**
   * Configures the Reference element, which specifies the content to sign
   * and and the digest method to use.
   */
  public void configureReference(SignerSignedInfo signedInfo, String xpathExpr)
  throws SignedInfoException,
         URIException,
         ReferenceException
  {
    // Create the reference
    SignerReference reference = signedInfo.createReference();

    // Set a 'null' URI in the reference.  This select the entire document.
    reference.setURI(new URI(""));

    // Insert an XPath transform, to select particular nodes to be signed
    setXPathTransform(reference, xpathExpr);

    // Insert an Enveloped Signature transform, so the <Signature> itself is not signed
    setEnvelopedSignatureTransform(reference);

    // Always insert a Decryption Transform, in case someone encrypts
    // portions of the signed content after we sign it.
    setDecryptionTransform(reference);

    // Specify the digest algorithm to use
    setDigestMethod(reference);

    // Add this Reference to the SignedInfo
    signedInfo.addReference(reference);
  }


  /**
   * Inserts an XPath transform in the reference provided.
   */
  public void setXPathTransform(SignerReference reference, String xPath)
  throws URIException,
         ReferenceException
  {
    if(xPath == null) return ;

    // Set an XPath transform to select a portion of the document
    TransformImplXPath xpathTransform = new TransformImplXPath();

    // Set the XPath
    xpathTransform.setXPath(xPath);

    // Set the XPath transform in the Reference
    reference.insertTransformAt(xpathTransform, reference.getTransformsNumber());
  }

  /**
   * Inserts an Enveloped-Signature Transform in the reference provided.
   */
  public void setEnvelopedSignatureTransform(SignerReference reference)
  throws URIException,
         ReferenceException
  {
    // Create an enveloped signature transform
    TransformImplEnvelopedSignature envTransform = new TransformImplEnvelopedSignature();

    // Set the enveloped signature transform in the Reference
    reference.insertTransformAt(envTransform, reference.getTransformsNumber());
  }

  /**
   * Inserts a Decryption Transform in the reference provided.
   */
  public void setDecryptionTransform(SignerReference reference)
  throws URIException,
         ReferenceException
  {
    // Create a decryption transform
    TransformImplDecryption decTransform = new TransformImplDecryption();

    // Set the decryption transform in this Reference
    reference.insertTransformAt(decTransform, reference.getTransformsNumber());
  }

  /**
   * Sets the canonicalization algorithm.
   */
  public void setCanonicalizationAlgorithm(SignerSignedInfo signedInfo)
  throws URIException,
         SignedInfoException
  {
    // Configure a canonicalization algorithm
    CanonicalizationAlgorithmImplCanonicalXML c14nAlg = new CanonicalizationAlgorithmImplCanonicalXML();

    // Set canonicalization algorithm
    signedInfo.setCanonicalizationAlgorithm(c14nAlg);
  }

  /**
   * Sets the digest algorithm for the signature to be SHA-1
   */
  private void setDigestMethod(SignerReference reference)
  throws URIException,
         ReferenceException
  {
    // Create a digest algorithm
    DigestAlgorithmImplSHA1 digestAlg1 = new DigestAlgorithmImplSHA1();

    // Set the digest algorithm
    reference.setDigestAlgorithm(digestAlg1);
  }
}
