//===========================================================================
//
// Copyright (c)  2002-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.xml.encrypt;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.security.cert.CertificateException;
import java.util.Vector;

import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import iaik.ixsil.exceptions.InitException;
import iaik.ixsil.init.IXSILInit;
import iaik.ixsil.util.URI;
import iaik.x509.X509Certificate;

import com.entrust.toolkit.KeyAndCertificateSource;
import com.entrust.toolkit.Trustmanager;
import com.entrust.toolkit.User;
import com.entrust.toolkit.examples.xml.utils.Utils;
import com.entrust.toolkit.exceptions.UserBadPasswordException;
import com.entrust.toolkit.exceptions.UserFatalException;
import com.entrust.toolkit.exceptions.UserNotLoggedInException;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.xencrypt.core.Decryptor;
import com.entrust.toolkit.xencrypt.core.Encryptor;
import com.entrust.toolkit.xencrypt.exceptions.DecryptorException;
import com.entrust.toolkit.xencrypt.exceptions.EncryptionHandlerException;
import com.entrust.toolkit.xencrypt.exceptions.XMLEInitException;
import com.entrust.toolkit.xencrypt.init.XMLEConstants;
import com.entrust.toolkit.xencrypt.init.XMLEInit;

/**
 * This class demonstrates how to use the Toolkit to encrypt and decrypt XML 
 * documents.
 * <p>
 * This is just a basic example in which the document has only one recipient and
 * a new EncryptedKey is created for every encrypted XML element.  To see how one
 * controls the creation of EncryptedKeys and how one encrypts for multiple
 * recipients, please refer to the EncryptExtended.java sample program.
 */
public class Encrypt {

    /**
    * The XMLE initialization object
    */
    public static XMLEInit m_initializer = null ;

    /**
    * The user that will validate encryption certificates when encrypting.
    */
    public static User m_sender = null ;

    /**
    * The recipient, who has the private decryption key
    */
    public static User m_recipient = null  ;

    /**
    * The Encryptor instance
    */
    public static Encryptor m_encryptor = null ;

    /**
    * Encryption certificate of recipient
    */
    public static X509Certificate m_certificate ;

    /**
     * Just for demonstration purposes, XML elements with this name will be encrypted.
     */
    private static final String ENCRYPT_FOR_RECIPIENT_A = "EncryptForA";

    /**
     * Just for demonstration purposes, the content of XML elements with this name will be encrypted.
     */
    private static final String ENCRYPT_CONTENT_FOR_RECIPIENT_A = "EncryptContentForA";

    /**
    * main method -- Log in, encrypt a document, decrypt the encrypted document.
    */
	public static void main(String args[])
	throws FileNotFoundException,
	       IOException,
	       CertificateException,
	       UserNotLoggedInException,
	       EncryptionHandlerException,
	       XMLEInitException,
	       InitException,
	       Exception
	{
	    // Get the command line arguments
        if (args.length != 6 && args.length != 7 )
        {
            System.out.println("\nPlease specify the following command line parameters:");

            System.out.println("  (1) Path and filename of an Entrust user profile representing the sender");
            System.out.println("  (2) Path and filename of an Entrust user profile representing the recipient");
            System.out.println("  (3) Profile password (same for both user profiles)");
            System.out.println("  (4) The URL of an initialization properties file\n\t" +
                                      "e.g. \"file:/C:/etjava/examples/xml/init/properties/init.properties\"\n\t" +
                                      "e.g. \"init.properties\"\n\t" +
                                      "e.g. \"http://hostname/init.properties\"");
            System.out.println("  (5) A file name for the encrypted XML document");
            System.out.println("  (6) A file name for the decrypted XML document");
            System.out.println("  (7) [Optional - an encryption certificate for the recipient in (2)]\n      " +
                                      "If none is provided, the sample will take it from the recipient's user profile");
            System.exit(0);
        }

	    int argsNumber = args.length ;

        int index = 0 ;
		String sender         = args[index++];
		String recipient      = args[index++];
		String password       = args[index++];
		String propertiesURL  = args[index++];
		String encryptedFile  = args[index++] ;
		String decryptedFile  = args[index++];

	    // Get the properties for the XMLE system.
        URI initProps = new URI(propertiesURL);

	    // Initialize IXSIL because XMLE invokes a few of its utility methods.
        System.out.println("Initializing IXSIL properties from \"" + initProps + "\"...");
        IXSILInit.init(initProps);

	    // Initialize XMLE.
        System.out.println("Initializing XMLE properties from \"" + initProps + "\"...");
        m_initializer = new XMLEInit(initProps);

		// Log in to the Toolkit
        m_sender = Utils.login(sender, new SecureStringBuffer(new StringBuffer(password)));

		// Log in to the Toolkit
        m_recipient = Utils.login(recipient, new SecureStringBuffer(new StringBuffer(password)));

        // Get the encryption certificate of a recipient
		if(args.length == 7) {
            m_certificate = new X509Certificate(new FileInputStream(args[index++]));
		}
		else {
            m_certificate = m_recipient.getEncryptionCertificate();
		}


        // This XML sample contains elements named 'EncryptForA'
		String documentToEncrypt  = "data/xml/encrypt/encryptForAandB.xml";


        // Encrypt
        encrypt(documentToEncrypt, encryptedFile) ;

        // Decrypt
        decrypt(encryptedFile, decryptedFile) ;

        // Done.
        m_sender.logout();
        m_recipient.logout();
	}

    /**
    * Encrypts the document and writes the result to a file.
    * @param xmlPlaintextFile The file name of an XML document to be encrypted
    * @param encryptedFile The file name of the encrypted XML document
    */
    public static void encrypt(String xmlPlaintextFile, String encryptedFile)
    throws EncryptionHandlerException,
           FileNotFoundException,
           UserNotLoggedInException,
           UserFatalException
    {

        System.out.println("Encrypting for: " + m_certificate.getSubjectDN().getName());

	    // Create an Encryptor instance
	    m_encryptor = new Encryptor(m_initializer, new FileInputStream(xmlPlaintextFile));

	    // Provide a trust manager, so encryption certificates are validated
        m_encryptor.setTrustmanager(new Trustmanager(new KeyAndCertificateSource(m_sender)));

	    // Set the symmetric encryption algorithm to be AES or omit this line
	    // to let the toolkit use its default.  Additional algorithm identifiers
	    // are found in the Javadoc for class XMLEConstants.
        m_encryptor.setSymmetricAlgorithm(XMLEConstants.ALGORITHM_AES_256);

	    // Specify a value for the Id attribute of EncryptedKey elements.  Omit
	    // this line to let the toolkit set those attributes to default values.
	    // (see the Javadoc for the Encryptor class).
        m_encryptor.setEncryptedKeyBaseID("EK");

	    // Specify a value for the Id attribute of EncryptedData elements.  Omit
	    // this line to let the toolkit set those attributes to default values.
	    // (see the Javadoc for the Encryptor class).
        m_encryptor.setEncryptedDataBaseID("ED");

        // As an example, encrypt all elements named "EncryptForA"
        getElements(ENCRYPT_FOR_RECIPIENT_A, false);

        // As an example, encrypt the content of all elements named "EncryptContentForA"
        getElements(ENCRYPT_CONTENT_FOR_RECIPIENT_A, true);

        // Encrypt the elements
        m_encryptor.encrypt();

        // Write the encrypted document
        System.out.println("Writing to: " + encryptedFile);
        m_encryptor.toOutputStream(new FileOutputStream(encryptedFile));
    }


    /**
     * Encrypts all DOM elements that have a particular tag name.
     *
     * This method uses the DOM to collect those elements.  In practice, your application will have
     * its own way of specifying the DOM elements you want to encrypt, the recipient(s) of each element,
     * and whether the element content or the whole element should be encrypted.
     *
     * @param elementTag The name of the elements to encrypt
     * @param contentOnly If 'true', only the element content is encrypted, not the element tag
     */
    private static void getElements(String elementTag, boolean contentOnly)
    throws EncryptionHandlerException
    {
        // Locate the DOM elements we would like to encrypt.
        NodeList elements = m_encryptor.getDocument().getElementsByTagName(elementTag);
        Element eles[] = new Element[elements.getLength()];
        for(int i = 0; i < elements.getLength(); i++){
            eles[i] = (Element) elements.item(i); 
        }
        
        for (int j = 0; j < eles.length; j++) {
            encryptElement(eles[j], contentOnly, m_certificate);
        }
    }


    /**
     * Encrypts one DOM element for one recipient.
     *
     * The contentOnly argument controls whether the whole element is encrypted or only
     * its content.  Please refer to the javadoc for the method setContentOnly(Element, boolean)
     * for further explanation.
     *
     * @param element The element to encrypt
     * @param contentOnly If 'true', only the element content is encrypted, not the element tag
     * @param certificate The encryption certificate of the recipient
     */
    public static void encryptElement(Element element, boolean contentOnly, X509Certificate certificate)
    throws EncryptionHandlerException
    {
        // Add a recipient to this element
        m_encryptor.setRecipient(element, certificate);

        // Specify the type of encryption, content only or element plus content
        m_encryptor.setContentOnly(element, contentOnly);

        // OPTIONAL:  If you prefer, one can encrypt the element immediately:
        /*
        if(contentOnly){
            m_encryptor.encryptContent(element);
            m_encryptor.updateContent(element);
        }
        else{
            m_encryptor.encryptElement(element);
            m_encryptor.updateElement(element);
        }
        */
        
    }

    /**
    * Decrypts the document and writes the result to a file.
    *
    * @param encryptedFile The file name of an encrypted XML document
    * @param decryptedFile The file name of the decrypted XML document
    */
    public static void decrypt(String encryptedFile, String decryptedFile)
    throws EncryptionHandlerException,
           FileNotFoundException,
           UserNotLoggedInException,
           UserBadPasswordException
    {

        Decryptor decryptor = null ;

	    System.out.println("Decrypting...");

        // Create a Decryptor instance
        decryptor = new Decryptor(m_initializer, new FileInputStream(encryptedFile));

        // Display recipient names
        showRecipients(decryptor);

        // Attach a user who can decrypt, based on the recipient identities
	    decryptor.addUser(m_recipient);

        // Decrypt any element that has that User as a recipient.
        decryptor.decrypt();

        // Write the decrypted document
        System.out.println("Writing to: " + decryptedFile);
        decryptor.toOutputStream(new FileOutputStream(decryptedFile));
    }

    /**
    * Displays recipient names.  This basic example has just one recipient.  In general
    * this method allows the application to identify which User instances have the
    * necessary private decryption keys for a particular document.  The application would
    * use that information to determine which User objects to attach to the Decryptor instance.
    * This simple example merely displays the recipients' names.
    *
    * @param decryptor The Decryptor instance
    */
    public static void showRecipients(Decryptor decryptor)
    throws DecryptorException,
           UserNotLoggedInException,
           UserBadPasswordException
    {
        Vector recipients = decryptor.getRecipients();
        for(int i=0; i<recipients.size(); i++) {
            System.out.println("\"" + recipients.elementAt(i) + "\"" + " is a recipient.");
        }

        // The user who will decrypt must be one these recipients, of course.
        System.out.println("\"" + m_recipient.getVerificationCertificate().getSubjectDN().getName() +
                           "\"" + " will decrypt.");
    }

}