//===========================================================================
//
// Copyright (c)  2001-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.xml.sign;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.PrivateKey;

import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.xml.sax.SAXException;

import iaik.ixsil.algorithms.CanonicalizationAlgorithmImplCanonicalXML;
import iaik.ixsil.algorithms.DigestAlgorithmImplSHA1;
import iaik.ixsil.algorithms.TransformImplEnvelopedSignature;
import iaik.ixsil.algorithms.TransformImplXSLT;
import iaik.ixsil.core.Position;
import iaik.ixsil.core.Signer;
import iaik.ixsil.core.SignerReference;
import iaik.ixsil.core.SignerSignature;
import iaik.ixsil.core.SignerSignedInfo;
import iaik.ixsil.exceptions.SignatureHandlerException;
import iaik.ixsil.init.IXSILInit;
import iaik.ixsil.keyinfo.SignerKeyManager;
import iaik.ixsil.util.ExternalReferenceResolverImpl;
import iaik.ixsil.util.URI;
import iaik.x509.X509Certificate;

import com.entrust.toolkit.KeyAndCertificateSource;
import com.entrust.toolkit.User;
import com.entrust.toolkit.examples.xml.utils.JAXParse;
import com.entrust.toolkit.examples.xml.utils.Utils;
import com.entrust.toolkit.util.SecureStringBuffer;

/**
 * This class shows how to create an enveloped XML signature that applies
 * an XSLT transform.  This sample displays on the console the XML content
 * that is being signed, reported as "Content of Reference 0.".  You will 
 * notice that this is the transformed content, not the original XML content
 * in the document.
 * <p>
 * The "examples\data\xml\sign" folder contain a simple SOAP message and an
 * XSL stylesheet you can use with it.  The stylesheet presents the XML content 
 * in a tabular format when you view the message file in a web browser.  
 * It is this tabular form that is signed by the sample (see Content of Reference 0.).
 * </p>
 * <p>
 * The same stylesheet applies to the presentation of the signed document.
 * For example, when viewed in a browser, the signer's name is displayed below
 * the table.
 * </p> 
 */
public class SignEnvelopedXSLT
{
  /**
   * A source of keys and certificates
   */
  private KeyAndCertificateSource m_source ;
  
  /**
   * Name of a file that is the XSLT stylesheet
   */
  private String m_stylesheetFileName = null ;


  /**
   * The main method.
   */
  public static void main(String[] args)
  throws Exception
  {
    SignEnvelopedXSLT signatureExample = new SignEnvelopedXSLT(args);

    // Optional -- display the XML resource to be signed.
    // Utils.displayURL(args[3]);

    // Find where to insert the Signature
    String insertSignatureHere = args[5] ;

    // Sign the document.
    Document signatureDOMDoc = signatureExample.createSignature(args[3], insertSignatureHere);

    // Write the signed document to a disk file
    Utils.writeDocumentToFile(args[4], signatureDOMDoc);

    // Optional -- Display signed document.
    Utils.displayFile(args[4]) ;

    System.out.println("Done");
  }

  /**
   * Constructor for this example. Logs into a User object or otherwise provide
   * a KeyAndCertificateSource, which contains a private signing key and 
   * a certificate.  
   */
  public SignEnvelopedXSLT(String[] args)
  throws Exception
  {
    if (args.length != 7)
    {
      System.out.println("\nPlease specify the following command line parameters:");
      System.out.println("  (1) Path and filename of an Entrust user profile,  e.g. \"data\\userdata\\RSAUser1.epf\"");
      System.out.println("  (2) Profile password, e.g. ~Sample7~");
      System.out.println("  (3) The URL of an IXSIL initialization properties file\n\t" +
                                "e.g. \"file:/C:/etjava/examples/xml/init/properties/init.properties\"\n\t" +
                                "e.g. \"init.properties\"\n\t" +
                                "e.g. \"http://hostname/init.properties\"");
      System.out.println("  (4) URI of an XML resource to be signed, e.g. \"file:data/xml/sign/message1.xml\"");
      System.out.println("  (5) Path and filename for the resulting XML signature file, e.g. \"data\\xml\\sign\\message1signed.xml\" ");
      System.out.println("  (6) Name of an XML element in (4) where the Signature is to be inserted, e.g. \"s:Body\"");
      System.out.println("  (7) File name of a stylesheet for the XSLT transform,  e.g. \"data\\xml\\sign\\message1.xsl\"");      
      System.exit(0);
    }
    
    // XSLT transform
    m_stylesheetFileName = args[6];

    // Need the signing key and certificate to be in a KeyAndCertificateSource wrapper.
    m_source = new KeyAndCertificateSource();
    
    // If you using a keystore such as P12, PFX, Entrust EPF, or java.security.KeyStore,
    // then log in here, to get access to its signing key and certificate.
    User user = Utils.login(args[0], new SecureStringBuffer(new StringBuffer(args[1])));
    PrivateKey pk = user.getSigningKey();
    X509Certificate cert = user.getVerificationCertificate();
    
    // In any case, provide a signing key and verification certificate from whatever source you use.
    m_source.setSigningInfo(pk, cert);
   
    // The properties file is specified by a URI.  You can provide the name of the
    // file, e.g 'init.properties', to the URI constructor.  In that case, the folder
    // that contains init.properties must be in the classpath.
    // Please refer to the comments in the 'init.properties' sample.
    URI initProps = new URI(args[2]);

    // Initialize IXSIL
    System.out.println("Initializing IXSIL properties from \"" + initProps + "\"...");
    IXSILInit.init(initProps);
    System.out.println("IXSIL is initialized.");
  }

  /**
   * Creates an enveloped signature with an XSLT transform.
   * @param uriTobeSigned The URI of an XML resource to be signed, e.g "file:/c:/test/tobesigned.xml"
   * @param insertSignatureHere The name of an XML element in the resource to be signed.  The Signature
   *                            will be inserted at that position.  If null, the signature will be inserted
   *                            at an arbitrary position, since this is just a simple example.
   * @return A DOM document that is the same as the resource to be signed but with a Signature element inserted.
   */
  public Document createSignature(String uriTobeSigned, String insertSignatureHere)
  throws Exception
  {
    // Define a position in the XML document where you want to insert the Signature.
    // To work with the SOAP sample and stylesheet provided, you would use "s:Body",
    // and the enveloped Signature element will be a child of the SOAP-ENV:Body.
    String xpath = "(//" + insertSignatureHere + ")[1]";

    System.out.println("Inserting dsig:Signature element at:" + xpath);
    Position signaturePosition = new Position(xpath, "", 0) ;

    URI baseURI = new URI(uriTobeSigned);
    ExternalReferenceResolverImpl res = new ExternalReferenceResolverImpl(baseURI);
    InputStream istrURI = res.resolve(baseURI);

    // Create a Signer, using the constructor that inserts a Signature
    // element into an existing document.
    //
    // Note: If your application reads the XML document directly from an InputStream,
    //       so there is no base URI, set the 'baseURI' parameter to 'null'.
    
    
    // URL to the noNamespaceSchemaLocation.  Required if the XML source does not declare 
    // an URL for the 'xsi:noNamespaceSchemaLocation' AND you must do a validating parse,
    // because you intend to sign a particular element in the document.
    // Note: If you are signing the whole document, you don't need a validating parse.
    String noNamespaceSchemaLocation = null;  // e.g. "file:foo.xsd"

    // Similarly, provide a sequence of namespace/URL pairs for any additional schemas
    // that are required for a successful validating parse.  The format is, e.g.
    //  "http://www.w3.org/2000/09/xmldsig# file:/etjava/examples/xml/init/schemasSignature.xsd"
    // Note 1: You do not actually need to set the XML dsig schema URL here.  The Toolkit will find
    //         its from the property 'DOMUtils.SignatureSchema' in the 'init.properties' file.
    //         Set the URLs for any other required namspaces.
    // Note 2: If you are signing the whole document, you don't need a validating parse.
    String schemaLocations = null;
    
    // Get a Signer
    Signer signer = createSigner(istrURI, baseURI, signaturePosition,
                                 noNamespaceSchemaLocation, schemaLocations);
 
    // Optional -- show where we will insert the Signature element.
    // The SignedInfo is empty so far.
    // displayDocument(signer.toDocument(), "");

    // Get an interface for the signature
    SignerSignature signature = signer.getSignature();

    // Get an interface for signed information
    SignerSignedInfo signedInfo = signer.getSignature().getSignerSignedInfo();

    // Set ID attribute of the signature.
    signature.setId("Signature001");

    // Configure a canonicalization algorithm
    CanonicalizationAlgorithmImplCanonicalXML c14nAlg = new CanonicalizationAlgorithmImplCanonicalXML();

    // Set canonicalization algorithm
    signedInfo.setCanonicalizationAlgorithm(c14nAlg);

    // Configure signature algorithm
    Utils.configureAlgorithm(m_source, signedInfo);

    // Create a reference
    SignerReference reference = signedInfo.createReference();

    // Configure the Reference for an enveloped signature, which must specify an enveloped signature transform

    // An empty URI signs the whole document, except for the <Signature> itself.
    URI tobeSigned = new URI("");
    
    // You can sign particular elements, identified by their ID attributes.
    // In this case, you must use a validating parse when creating the Signer,
    // so you might have to provide a noNamespaceSchemaLocation and set some
    // schemaLocations as above, unless the document itself declares their URLs.
    //
    // String id = "foobar1";  // not "#foobar1" 
    // URI tobeSigned = new URI(null, null, "", null, id);
    
    // Set the URI into this <dsig:Reference>
    reference.setURI(tobeSigned);

    // Create an enveloped signature transform
    TransformImplEnvelopedSignature transform = new TransformImplEnvelopedSignature();

    // Set the enveloped signature transform in the Reference, so the signing
    // operation will not sign the Signature element itself
    reference.insertTransformAt(transform, 0);
    
    // Optional: Create an XSLT transform.  In the trace of this sample,
    // you'll see that "Content of Reference 0" over which the signature is
    // applied is the *transformed* XML, not the original content.
    TransformImplXSLT xslt = new TransformImplXSLT();
    xslt.setStyleSheet(new FileInputStream(m_stylesheetFileName), null);
    reference.insertTransformAt(xslt, reference.getTransformsNumber());    

    // Create a digest algorithm
    DigestAlgorithmImplSHA1 digestAlg1 = new DigestAlgorithmImplSHA1();

    // Set the digest algorithm
    reference.setDigestAlgorithm(digestAlg1);

    // Add the Reference to the SignedInfo
    signedInfo.addReference(reference);

    // Recommended -- Attach the user's X.509 verification certificate to the Signature
    int x509Data = Utils.X509DATA_CERTIFICATE;

    // Optional -- Attach the X.509 certificate's subject name and/or issuer and serial number
    x509Data |= Utils.X509DATA_SUBJECT_NAME | Utils.X509DATA_ISSUER_SERIAL;

    // Set the X509Data
    SignerKeyManager keyManager = Utils.setX509KeyInfo(x509Data,
                                                        signer,
                                                        m_source.getVerificationCertificate(),
                                                        "KeyInfo001");
    // For MS .NET.
    //
    // The XML dsig verifier in .NET (the .NET Framework SignedXml class) does not
    // recognize X.509 certificates, so the signer must attach raw <KeyValue> in
    // place of <X509Data>.  However, a signature can be validated on the basis
    // of <KeyValue> only if the verifier already possesses a trusted copy of the
    // signer's key.  Transporting a public key via a <KeyValue> element is not
    // secure (even if the <KeyInfo> is signed), so your verifier must take
    // precautions -- please read the javadoc for the methods 'setKeyValue(...)'
    // and especially 'validatePublicKey(...)' in 'examples/xml/utils/Utils.java'.
    //
    // If you need to support MS .NET, use the following in place of setX509KeyInfo(...) above.
    // SignerKeyManager keyManager = Utils.setKeyValue(signer,
    //                                        m_user.getVerificationCertificate().getPublicKey(),
    //                                        "KeyInfo001");

    // Optional -- Sign the KeyInfo element
    Utils.signKeyInfo(signedInfo, keyManager);

    // Apply the transforms in each Reference and ensure the resulting XML content is not empty.
    Utils.processTransforms(signature, true /* Optional: display the content that will be signed */);

    // Compute the signature value
    signer.getSignature().sign();

    // Optional, display the document
    // displayDocument(signer.toDocument(), "");

    // Get resulting XML document
    return signer.toDocument();
  }
  
  /**
   * Gets a Signer that can create an enveloped signature.
   * @param istrURI an InputStream to XML source
   * @param baseURI the base URI for the XML source
   * @param signaturePosition the Position in the document where the enveloped signature will be placed
   * 
   * @return Signer can be used to sign the document
   */
  private Signer createSigner(InputStream istrURI, URI baseURI, Position signaturePosition,
                              String noNamespaceSchemaLocation, String schemaLocations)
  throws  SignatureHandlerException,
          IOException,
          SAXException,
          ParserConfigurationException
  {
      Signer signer = null;
      
      // Demonstrate two ways to get a Signer, from an existing Document instance, or from an InputStream.
      final boolean USEJAXP = false;
      if (USEJAXP)
      {
          // Method (1)       
          // Your application might already have parsed existing XML source to a Document
          // and perhaps manipulated it. In that case, you can use the Signer constructor that 
          // takes a Document instance, so the document in not parsed over again by the Toolkit.
          JAXParse parser = new JAXParse(noNamespaceSchemaLocation, schemaLocations);
          Document doc = parser.parse(istrURI);
          signer = new Signer(doc, baseURI, signaturePosition);
      }
      else
      {
          // Method (2)
          // The Signer class can parse the existing XML source.
          signer = new Signer( istrURI, baseURI, signaturePosition,
                               noNamespaceSchemaLocation, schemaLocations );
      }
      return signer;
  }
}

