//===========================================================================
//
// Copyright (c)  2001-2024 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.xml.utils;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.net.URL;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.interfaces.DSAPublicKey;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.xml.sax.SAXException;

import iaik.asn1.structures.Name;
import iaik.ixsil.algorithms.DigestAlgorithmImplSHA1;
import iaik.ixsil.algorithms.SignatureAlgorithm;
import iaik.ixsil.algorithms.SignatureAlgorithmImplDSA;
import iaik.ixsil.algorithms.SignatureAlgorithmImplECDSA;
import iaik.ixsil.algorithms.SignatureAlgorithmImplRSA;
import iaik.ixsil.core.Signer;
import iaik.ixsil.core.SignerReference;
import iaik.ixsil.core.SignerSignature;
import iaik.ixsil.core.SignerSignedInfo;
import iaik.ixsil.core.Verifier;
import iaik.ixsil.core.VerifierSignature;
import iaik.ixsil.exceptions.DOMUtilsException;
import iaik.ixsil.exceptions.KeyProviderException;
import iaik.ixsil.exceptions.ReferenceException;
import iaik.ixsil.exceptions.SignatureException;
import iaik.ixsil.exceptions.SignedInfoException;
import iaik.ixsil.exceptions.URIException;
import iaik.ixsil.init.IXSILConstants;
import iaik.ixsil.init.IXSILInit;
import iaik.ixsil.keyinfo.KeyManagerImpl;
import iaik.ixsil.keyinfo.KeyProviderImplKeyValue;
import iaik.ixsil.keyinfo.KeyProviderInterface;
import iaik.ixsil.keyinfo.SignerKeyManager;
import iaik.ixsil.keyinfo.x509.KeyProviderImplX509Data;
import iaik.ixsil.keyinfo.x509.X509Data;
import iaik.ixsil.keyinfo.x509.X509IssuerSerial;
import iaik.ixsil.keyinfo.x509.X509SubjectName;
import iaik.ixsil.keyinfo.x509.X509TrustManagerInterface;
import iaik.ixsil.util.DOMUtils;
import iaik.ixsil.util.URI;
import iaik.utils.Base64InputStream;
import iaik.utils.Base64OutputStream;
import iaik.utils.RFC2253NameParserException;
import iaik.x509.X509Certificate;

import com.entrust.toolkit.KeyAndCertificateSource;
import com.entrust.toolkit.Trustmanager;
import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.credentials.PKCS12Reader;
import com.entrust.toolkit.exceptions.UserFatalException;
import com.entrust.toolkit.exceptions.UserNotLoggedInException;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.xml.dsig.keyinfo.tokenref.KeyProviderImplSecurityTokenRef;


/**
 * This class has static methods used by the XML digital signature and
 * XML encryption examples, collected here for convenience.
 */
public class Utils
{
  /**
   * The signer will put his verification certificate in the signature.
   */
    public static final int X509DATA_CERTIFICATE   = 1;

  /**
   * The signer will put his certificate's subject name in the signature.
   */
    public static final int X509DATA_SUBJECT_NAME  = 2;

  /**
   * The signer will put his certificate's issuer name and serial number in the signature.
   */
    public static final int X509DATA_ISSUER_SERIAL = 4;
    
        
  /**
   * Some of the Java character encodings supported in Sun's JVM. 
   * Sun's i18n.jar and JDK 1.4 supports Big5 and others.
   */
    private final static String javaEncodings[] = { "UTF8", "UTF16",
         "UnicodeBigUnmarked", "UnicodeLittleUnmarked", "ISO8859_1", "ASCII", "Big5" };    
  /**
   * Character encoding names used in XML declaration, in the same order as 'javaEncodings'.
   */
    private final static String xmlEncodings[] = { "UTF-8", "UTF-16",
        "UTF-16BE",  "UTF-16LE", "ISO-8859-1", "US-ASCII", "Big5" };    

  /**
   * Logs in to an Entrust user profile (.epf) or a PKCS#12 file (.p12 or .pfx).
   *
   * @param profile The name of the Entrust user profile or PKCS#12 file (e.g. "user.epf or user.pfx").
   * @param password The password.
   */
  public static User login(String profile, SecureStringBuffer password)
  throws Exception
  {
    try {
        System.out.println("Logging in to: " + profile);

        CredentialReader credreader = null ;

        if(profile.endsWith(".epf")) {
            credreader = new FilenameProfileReader(profile);
        }
        else if (profile.endsWith(".p12") || profile.endsWith(".pfx")) {
            credreader = new PKCS12Reader(new FileInputStream(profile));
        }

        User user = new User() ;
        user.login(credreader, password);
        return user ;
    }
    catch (Exception e) {
        throw new Exception("Error logging in: " + e.getMessage());
    }
  }

 /**
   * Sets a SignatureAlgorithm of the correct kind, depending on whether
   * this user has an RSA or DSA private signing key.
   *
   * @param source a KeyAndCertificateSource that must be logged in
   * @param signedInfo The SignerSignedInfo for the XML signature
   */
  public static void configureAlgorithm(KeyAndCertificateSource source, SignerSignedInfo signedInfo)
  throws Exception
  {
    // Create a signature algorithm of the appropriate kind, which
    // depends on the kind of private key this user possesses.

    PrivateKey privatekey = source.getSigningKey();

    SignatureAlgorithm alg = null;
    if(privatekey.getAlgorithm().equals("DSA"))
    {
        alg = (SignatureAlgorithm) new SignatureAlgorithmImplDSA();
    }
    else if(privatekey.getAlgorithm().equals("RSA"))
    {
        alg = (SignatureAlgorithm) new SignatureAlgorithmImplRSA();
    }
    else if(privatekey.getAlgorithm().equals("EC"))
    {
        alg = (SignatureAlgorithm) new SignatureAlgorithmImplECDSA();;
    }
    else
    {
        throw new Exception("Unsupported algorithm: " + privatekey.getAlgorithm());
    }
    
    // Set private key
    alg.setSignerKey(privatekey);

    // Set signature algorithm
    signedInfo.setSignatureAlgorithm(alg);

    System.out.println("Signing with " + privatekey.getAlgorithm() + " private key");
  }
  


  /**
   * Inserts into an XML Signature a KeyInfo element that contains X509 data.
   * The X509Data element will contain either the X.509 verification certificate
   * of the signer or so-called "hints" that the Verifier could use to find that
   * certificate.  The hints consist of the certificate's subject name, or
   * its issuer and serial number.<p>
   *
   * If the signer includes a valid certificate, then these hints are redundant,
   * because the Toolkit's Verifier class does not use them.  However, you might
   * need to include them for interoperability with other dsig implementations.</p><p>
   *
   * If the KeyInfo element provides no explicit X509Certificate to the Verifier,
   * its trust manager will search for a verification certificate that matches the
   * "hints", from amongst the certificates that your verifier application gave it
   * when it called <code>com.entrust.toolkit.Trustmanager.putCertificates</code>.
   * </p>
   * <p>
   * <B>Note: Your application should sign the KeyInfo element if it will support
   *          non-repudiation of XML signatures.</B>
   * </p><p>
   *
   * <p>
   * Note: If your signer application provides X509Data, you can prevent your
   *          verifier application from accepting KeyValue information.   Edit
   *          'keymanager.properties' and remove this line;
   *
   *    <pre>
   *       Subelement.02 = http://www.w3.org/2000/09/xmldsig#:KeyValue
   *    </pre>
   * </p>
   *
   * @param x509Info The kind of X509Data to include
   * @param signer A Signer that represents this signature.
   * @param certificate The signer's verification certificate
   * @param keyInfoId A value for the optional ID attribute of the KeyInfo element,
   *           or null if the KeyInfo element will not be assigned an ID
   *
   * @see #X509DATA_CERTIFICATE
   * @see #X509DATA_SUBJECT_NAME
   * @see #X509DATA_ISSUER_SERIAL
   *
   */
  public static SignerKeyManager setX509KeyInfo(int x509Info,
                                Signer signer,
                                X509Certificate certificate,
                                String keyInfoId)
  throws KeyProviderException,
         RFC2253NameParserException,
         SignatureException,
         ReferenceException,
         SignedInfoException
  {
    Document signatureDOMDoc = signer.toDocument();

    // We will create a new KeyInfo element
    KeyManagerImpl keyManager = new KeyManagerImpl(signatureDOMDoc);
    SignerKeyManager signerKeyManager = (SignerKeyManager) keyManager;

    // Give this KeyInfo element an ID attribute.
    keyManager.setId(keyInfoId);

    // Create and configure a KeyInfo provider for the X509Data element
    KeyProviderImplX509Data keyProviderX509Data = new KeyProviderImplX509Data(signatureDOMDoc);

    X509Data x509 = new X509Data();

    if( (x509Info & X509DATA_CERTIFICATE) == X509DATA_CERTIFICATE) {
        x509.insertHintAt(certificate, x509.getHintNumber())  ;
    }

    if( (x509Info & X509DATA_SUBJECT_NAME) == X509DATA_SUBJECT_NAME) {
        setX509SubjectName(x509, certificate);
    }

    if( (x509Info & X509DATA_ISSUER_SERIAL) == X509DATA_ISSUER_SERIAL) {
        setX509IssuerSerial(x509, certificate);
    }

    // Put this X509 data into the key provider.
    keyProviderX509Data.insertX509DataAt(x509, keyProviderX509Data.getX509DataNumber());

    // Set the key provider into the key manager.
    keyManager.addKeyProvider(keyProviderX509Data);

    // Set the key manager into the signature.
    signer.getSignature().setKeyManager(signerKeyManager);

    return signerKeyManager ;
  }

  /**
   * Helper method that puts the signer's X509SubjectName element in the X509Data.
   *
   * @param x509 X509Data into which the X509SubjectName will be placed.
   * @param certificate The signer's verification certificate
   *
   * @see #setX509KeyInfo(int X509Data, Signer signer, X509Certificate certificate, String keyInfoId)
   *
   */
  public static void setX509SubjectName(X509Data x509, X509Certificate certificate)
  throws KeyProviderException,
         RFC2253NameParserException
  {
    X509SubjectName subjectName = new X509SubjectName(((Name)certificate.getSubjectDN()).getRFC2253String()) ;
    x509.insertHintAt(subjectName, x509.getHintNumber())  ;
  }

  /**
   * Helper method that puts an X509IssuerSerial element in the X509Data.
   *
   * @param x509 X509Data into which the X509IssuerSerial element will be placed.
   * @param certificate The signer's verification certificate
   *
   * @see #setX509KeyInfo(int X509Data, Signer signer, X509Certificate certificate, String keyInfoId)
   */
  public static void setX509IssuerSerial(X509Data x509, X509Certificate certificate)
  throws KeyProviderException,
         RFC2253NameParserException
  {
    String issuerName = ((Name)certificate.getIssuerDN()).getRFC2253String();
    X509IssuerSerial issuerserial = new X509IssuerSerial(issuerName, certificate.getSerialNumber()) ;
    x509.insertHintAt(issuerserial, x509.getHintNumber())  ;
  }

  /**
   * Inserts into an XML Signature a KeyInfo element that contains a KeyValue.
   * The KeyValue contains a key that could be used to verify the signature.
   *
   * <p>
   * <B>Note: If the signer includes X509Data in the signature, it is not necessary
   *          to attach a KeyValue too, and the method <CODE>setX509KeyInfo</CODE>,
   *          is preferred over <CODE>setKeyValue</CODE>.  However, you might include
   *          a KeyValue if your application does not use X.509 certificates at all.
   * </B>
   * </p>
   *
   * @param signer A Signer that represents this signature.
   * @param key The signer's public verification key
   * @param keyInfoId a value for the optional ID attribute of the KeyInfo element
   *           or null if the KeyInfo element will not be assigned an ID
   *
   * @see #isValidKeyValueInfo(PublicKey key)
   * @see #setX509KeyInfo(int X509Data, Signer signer, X509Certificate certificate, String keyInfoId)
   */
  public static SignerKeyManager setKeyValue(Signer signer,
                                 java.security.PublicKey key,
                                 String keyInfoId)
  throws KeyProviderException,
         RFC2253NameParserException,
         SignatureException,
         ReferenceException,
         SignedInfoException
  {
    // Instantiate a key manager
    Document signatureDOMDoc = signer.toDocument();
    KeyManagerImpl keyManager = new KeyManagerImpl(signatureDOMDoc);

    // Give the KeyInfo element an ID attribute.
    keyManager.setId(keyInfoId);

    // Create and configure a KeyInfo provider for a KeyValue clause
    KeyProviderImplKeyValue keyProviderKeyValue = new KeyProviderImplKeyValue(signatureDOMDoc);
    keyProviderKeyValue.setVerifierKey(key);

    // Set this KeyInfo provider into the key manager.
    keyManager.addKeyProvider(keyProviderKeyValue);

    // Set the key manager into the signature.
    SignerKeyManager skm = (SignerKeyManager) keyManager;
    signer.getSignature().setKeyManager(skm);

    return skm ;
  }
  
  /**
   * Inserts a KeyInfo element that contains a WS-Security SecurityTokenReference element into signature.
   *
   * @param signer Signer object for retrieving the signature.
   * @param x509Cert Signer's verification certificate.
   * @param keyInfoId Optional ID attribute of KeyInfo element.  If null, ID attribute for KeyInfo element is not included.
   *
   * @return a SignerKeyManager
   *
   * @throws SignatureException if IXSIL throws a SignatureException when setting the signerKeyManager into the Signature
   */
  public static SignerKeyManager setSecurityTokenReferenceKeyInfo(Signer signer, X509Certificate x509Cert, String keyInfoId)
  throws SignatureException
  {
      // key manager
      Document signatureDOMDoc = signer.toDocument();
      KeyManagerImpl keyManager = new KeyManagerImpl(signatureDOMDoc);
      
      // KeyInfo provider for SecurityTokenReference
      KeyProviderImplSecurityTokenRef keyProviderSecurityTokenRef = new KeyProviderImplSecurityTokenRef(signatureDOMDoc, x509Cert);
      keyProviderSecurityTokenRef.setVerifierKey(x509Cert.getPublicKey());
      
      // add key provider to key manager
      keyManager.addKeyProvider(keyProviderSecurityTokenRef);
      
      // set ID attribute of KeyInfo element
      keyManager.setId(keyInfoId);
      
      // set signature's key manager
      SignerKeyManager signerKeyManager = (SignerKeyManager)keyManager;
      signer.getSignature().setKeyManager(signerKeyManager);
      
      return signerKeyManager;
    }  

  /**
   * Creates a dsig Reference to the KeyInfo element, so it will be signed.
   *
   * <p>
   * <B>Note: Your application should sign the KeyInfo element if it will support
   *          non-repudiation of XML signatures.</B>   A problem would arise if
   *          the X509Data is not within the scope of the signature, and two certificates
   *          exist for the same public verification key.  For example, two certificates
   *          could exist with different policy, issued by different CAs.  If the X.509 certificate
   *          (or a unique identifier) is not within the signature's scope, there
   *          is ambiguity as to which certificate (and thus which policy, etc.) to use
   *          in verifying the signature.  In some circumstances, this ambiguity may be
   *          enough to remove any support for non-repudiation which would exist.
   * </p>
   *
   * <p>
   * <B>Note:</B> When the verification public key comes from a KeyValue element,
   *          binding the KeyInfo to the signature by signing it does not
   *          prevent any additional attack that is possible when the KeyInfo
   *          is not signed.
   * </p>
   *
   *
   * @param signedInfo A SignerSignedInfo that represents this signature.
   * @param keyManager The SignerKeyManager that represents the KeyInfo
   *
   * @see #setX509KeyInfo(int X509Data, Signer signer, X509Certificate certificate, String keyInfoId)
   * @see #isValidKeyValueInfo(PublicKey key)
   */
  public static void signKeyInfo(SignerSignedInfo signedInfo,
                                 SignerKeyManager keyManager)
  throws ReferenceException,
         SignedInfoException
  {
    String keyInfoId = keyManager.getId();
    if (keyInfoId == null)
    {
        // The application has not given the KeyInfo element an ID.
        throw new RuntimeException("You cannot sign a KeyInfo that has no ID attribute. "
                                 + "Call KeyManagerImpl.setId to give the KeyInfo an ID.");
    }

    // Create a reference
    SignerReference ref = signedInfo.createReference();

    // Set its URI to be the KeyInfo element
    ref.setURI(new URI(null, null, "", null, keyInfoId));

    // Create new instance of digest algorithm
    DigestAlgorithmImplSHA1 digestAlg1 = new DigestAlgorithmImplSHA1();

    // Set digest algorithm
    ref.setDigestAlgorithm(digestAlg1);

    // Add reference to signature
    signedInfo.addReference(ref);
  }

  /**
   * Process all transforms in all References, and throw an exception if
   * the resulting content to be signed is empty.  Otherwise, display the
   * content that will be signed.
   *
   * @param signature A SignerSignature that has all its References completely configured
   * @param display should be set 'true' to display the content that will be signed
   *
   * @exception Exception if any reference provides no content to sign
   */
  public static void processTransforms(SignerSignature signature, boolean display)
  throws Exception
  {
    // Process all Transforms in all References
    signature.computeDigestInputs();

    // Get the references
    SignerReference[] references = signature.getSignerSignedInfo().getSignerReferences();
    if(references.length == 0) {
        throw new Exception("There is no content to sign !");
    }

    // Check that the result of the transforms is not empty !
    for (int i = 0; i < references.length; i++)
    {
        ByteArrayInputStream bais =
            (ByteArrayInputStream)references[i].getExplicitData(SignerReference.EXPLICITDATATYPE_DIGESTINPUT_);
        if(bais == null || (bais.available() == 0)) {
            throw new Exception("Reference \"" + references[i].getURI() + "\" did not provide any content to sign !");
        }

        // Optional: Display the actual content that will be signed, after all transforms have been applied.
        //           Remove this in production code, especially if signing very large files.
        if(display) {
            
            // Display in the default character encoding on this platform.
            InputStreamReader reader = new InputStreamReader(bais, IXSILConstants.JAVA_ENCODING_UTF8_);
            String header = "----------- Content of Reference " + i + ".";
            System.out.println(header + " (Start) -----------");
            int r ;
            while ((r = reader.read()) != -1) {
                System.out.print((char) r);
            }
            System.out.println("\n" + header + " (End) -----------\n");

            // Must reset stream after reading it for display.
            bais.reset();
        }
    }
  }


  /**
   * Serializes a DOM document and writes it to a file.
   *
   * @param file The name of the file.
   * @param document The DOM document.
   */
  public static void writeDocumentToFile(String file, Document document)
  throws FileNotFoundException,
         IOException
  {
    FileOutputStream os = new FileOutputStream(file);
    serializeDocument(document, os);
  }
  
  /**
   * Returns a JAXP DocumentBuilder object, which can parse XML source into a DOM Document.
   * <p>
   * You can request a specific JAXP implementation by setting a system property, or
   * by adding a line in the 'jaxp.properties' file in your JRE lib folder, if any.  
   * If no specific JAXP implementation is requested, this method will return the 
   * implementation found in JAR files on the classpath, or in the JRE runtime itself.  
   * If there is no implementing class there, this method returns an internal 
   * implementation, which comes from Apache Xerces version 1.4.4.
   *  
   * </p><p>
   * Please refer to the javadoc for <CODE>javax.xml.parsers.DocumentBuilder.newInstance()</CODE>
   * for details on how you specify a JAXP implementation.
   * </p>
   * @return javax.xml.parsers.DocumentBuilder
   */
  public static DocumentBuilder getDocumentBuilder()
  throws ParserConfigurationException
  {
    // Optional: e.g. specify a particular JAXP provider by a system property.
    // System.setProperty("javax.xml.parsers.DocumentBuilderFactory", 
    //                    "org.apache.xerces.jaxp.DocumentBuilderFactoryImpl");
    
    // Get a javax.xml.parsers.DocumentBuilderFactory
    DocumentBuilderFactory dbFactory = DocumentBuilderFactory.newInstance();
    
    // Optional: While debugging, report which JAR file has the JAXP implementation
    System.out.println("JAXP implementation: \"" + 
                            getUrlToResource(dbFactory.getClass()) + "\"");
    
    dbFactory.setNamespaceAware(true);
    dbFactory.setIgnoringElementContentWhitespace(true);
    dbFactory.setExpandEntityReferences(false);
            
    // If parsing with validation, you must attach an ErrorHandler
    dbFactory.setValidating(false);
            
    // Check that the provided parser will be namespace aware
    DocumentBuilder db = dbFactory.newDocumentBuilder();  
    if(!db.isNamespaceAware()) {
        throw new ParserConfigurationException("The XML parser is not namespace aware !");
    }
    return db;
  }
  
  
  /**
   * Resolves an URL, confirms that the resource can be parsed as XML, and displays it.
   * @param url An URL that points to an XML resource
   */
  public static void displayURL(String url)
  throws DOMUtilsException,
         IOException,
         SAXException,
         ParserConfigurationException
  {
        displayDocument(getDocumentBuilder().parse(url), url);
  }

  /**
   * Confirms that a file can be parsed as XML and, if so, displays the XML document.
   * @param filename A file containing XML
   */
  public static void displayFile(String filename)
  throws DOMUtilsException,
         IOException,
         SAXException,
         ParserConfigurationException
  {
        displayDocument(getDocumentBuilder().parse(new FileInputStream(filename)), filename);
  }


  /**
   * Displays a DOM document.
   * @param document A DOM document
   *
   * @param insert An arbitrary string to insert around the start and end of the display
   */
  public static void displayDocument(Document document, String insert)
  throws IOException
  {
    // Serialize the DOM document into a UTF8 character encoding.
    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    serializeDocument(document, baos);
    
    System.out.print("\n--------- START OF " + insert +
                                " SERIALIZED DOCUMENT ---------\n" );
                                
    // Display in the default character encoding on this platform.
    ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
    InputStreamReader reader = new InputStreamReader(bais, IXSILConstants.JAVA_ENCODING_UTF8_);
    int r ;
    while ((r = reader.read()) != -1) {
        System.out.print((char) r);
    }
    
    System.out.println( "\n--------- END OF " + insert +
                                " SERIALIZED DOCUMENT ---------\n"  );
    baos.close();
    bais.close();
  }
  
  /**
   * Generates string representation of a DOM document.
   * 
   * @param doc is a DOM document to seriazn InputStream that provides the data.
   *                When this method returns, the input stream has been closed.
   * @return a String that is the serialized document
   * @exception IOException if the data cannot be serialized and encoded into UTF-8
   */
  public static String documentToString( Document doc )
  throws IOException {
    
    // Serialize the DOM document into a UTF8 character encoding.
    ByteArrayOutputStream bos = new ByteArrayOutputStream();
    serializeDocument(doc, bos );
    return bos.toString(IXSILConstants.JAVA_ENCODING_UTF8_);
  }  

  /**
   * Reads data from an input stream, compresses the data, base-64 encodes
   * that compressed data, and returns an output stream that holds the result.
   *
   * @param istream is an InputStream that provides the data.
   *                When this method returns, the input stream has been closed.
   * @return a ByteArrayOutputStream with the compressed, Base64 encoded data.
   * @exception IOException if the data cannot be compressed and Base 64 encoded.
   */
  public static ByteArrayOutputStream readBinary(InputStream istream)
  throws IOException
  {
    BufferedInputStream bis = new BufferedInputStream(istream);
    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    GZIPOutputStream out = new GZIPOutputStream(new Base64OutputStream(baos));
    byte[] temp = new byte[1024];
    while (true) {
      int bytesRead = bis.read(temp);
      if (bytesRead == -1) break;
      out.write(temp, 0, bytesRead);
    }
    bis.close();
    istream.close();
    out.close();
    return baos;
  }

  /**
   * Base-64 decodes a byte array, uncompresses that decoded data,
   * and writes the uncompressed data to the output stream provided.
   *
   * @param ostream the OutputStream to which the data should be
   *                written.  When the method returns, the stream
   *                has been closed.
   * @param in The byte array
   */
  public static void writeBinary(OutputStream ostream, byte[] in)
  throws IOException
  {
    BufferedOutputStream bos = new BufferedOutputStream(ostream);
    GZIPInputStream is = new GZIPInputStream(
                           new Base64InputStream(
                             new ByteArrayInputStream(in)));
    byte[] temp = new byte[1024];
    while (true) {
        int bytesRead = is.read(temp);
        if (bytesRead == -1) break;
        bos.write(temp, 0, bytesRead);
    }
    bos.flush();
    bos.close();
    ostream.close();
  }

  /**
  * Sets a trust manager on all X509Data in the provided Signature.  The Toolkit
  * validates the verification certificate before using its public key.
  *
  * @param verifier A Verifier
  * @param source  A KeyAndCertificateSource that contains a <code>CertVerifier</code>
  */
  public static void setTrustmanager(Verifier verifier, KeyAndCertificateSource source)
  throws UserNotLoggedInException,
          UserFatalException,
          SignatureException
  {
    X509TrustManagerInterface trustManager = new Trustmanager(source) ;

    // Set the trust manager on all X509Data and TokenRef data
    KeyProviderInterface[] providers = ((KeyManagerImpl)verifier.getSignature().getKeyManager()).getKeyProviders();
    for(int i=0; i<providers.length ; i++) {
        if(providers[i] instanceof KeyProviderImplX509Data) {
            ((KeyProviderImplX509Data)providers[i]).setTrustManager(trustManager);
        }
        else if(providers[i] instanceof KeyProviderImplSecurityTokenRef) {
            ((KeyProviderImplSecurityTokenRef)providers[i]).setTrustManager(trustManager);
        }
    }
  }
  


  /**
   * Determines whether a verification public key came from a valid certificate
   * or merely a KeyValue element in the XML Signature.  Please refer to the
   * javadoc in Utils.isValidKeyValueInfo(PublicKey key) for more info.<p>
   *
   * Pre-condition -- Your application should invoke this method after calling
   *                  VerifierSignature.verify().  Also, setTrustmanager()
   *                  must have been called before the signature was verified.
   *                  Please refer to the Verify.java sample.
   *
   * @param signature a VerifierSignature
   * @see #isValidKeyValueInfo(PublicKey key)
   */
  public static void validatePublicKey(VerifierSignature signature)
  throws Exception
  {
    // If the verification public key came from X509Data, then it was already
    // validated by IXSIL, because a TrustManager was set on all X509Data.
    KeyProviderInterface kp = signature.getKeyManager().getKeyProvider();
    if(kp instanceof KeyProviderImplX509Data || kp instanceof KeyProviderImplSecurityTokenRef)  {

        System.out.println("The signature was verified using a valid certificate.");
        return ;
    }

    // Public key came from a KeyValue, because the X509Data did not provide a valid certificate
    if(!isValidKeyValueInfo( (PublicKey)signature.getKeyManager().getVerifierKey() )) {
        throw new Exception("Failed to validate the KeyValue information");
    }
  }

  /**
   * Returns true if the public key from an XML signature's KeyValue element will be trusted.
   *
   * <DL><DT>Note:</DT>
   *    <DD>
   *    Unless you need to verify XML signatures on the basis of KeyValue
   *    info, it is recommended that you prevent the Verifier from accepting
   *    KeyValue info, by editing 'keymanager.properties'.
   *
   *    Remove this line, to ensure that the public key with which the
   *     Toolkit verified the signature came from a valid certificate;
   *
   *    <pre>
   *       Subelement.02 = http://www.w3.org/2000/09/xmldsig#:KeyValue
   *    </pre>
   *
   *    Otherwise, you must provide an implementation of <code>isValidKeyValueInfo(PublicKey)</code>
   *    which specifies your procedure for validating a public key received in
   *    a KeyValue element.<p>
   *
   *    For example, your verifier application might already have a copy of
   *    the signer's public key in a secure keystore, and it would check that
   *    the key provided to <code>isValidKeyValueInfo(PublicKey)</code> is
   *    identical to that trusted copy.  In this way, the verifier would
   *    associate the public key with a particular entity.
   *    </p>
   *
   *    </DD>
   * </DL>
   *
   * @param key is a PublicKey that came from a KeyValue, not a certificate
   * @return false if the public key is not valid
   *
   * @see #validatePublicKey(VerifierSignature signature)
   */
  public static boolean isValidKeyValueInfo(PublicKey key)
  {
    // Your validation procedure goes here -- it must return false
    // if the public key cannot be validated.

    // This is merely an example, so just display the verification key
    // and report that it was not validated.
    System.out.println("The signature was verified by the following public key: "
                       + " Because it came from the Signature's KeyValue element, "
                       + " this verification has no cryptographic significance.\n");

    if(key instanceof RSAPublicKey) {
        System.out.println(((RSAPublicKey) key).toString());
    }
    else if (key instanceof DSAPublicKey) {
        System.out.println(((DSAPublicKey) key).toString());
    }
    else if(key instanceof ECPublicKey) {
        System.out.println(((ECPublicKey) key).toString());
    }

    // Since no implementation is provided, accept public keys only
    // from valid certificates -- never accept KeyValue info.
    return false ;
  }
  
  /**
   * Creates a DOM Document instance using the JAXP API.  JAXParse is simple wrapper 
   * around the required JAXP methods.  The returned Document instance can be passed 
   * directly into the Toolkit's Signer or Verifier constructor.
   *
   *@see #initSignatureSchemaLocation()
   */
  public static Document createDOMDocumentJAXP(InputStream istream, String noNamespaceSchemaLocation, String schemaLocations)
  throws ParserConfigurationException,
         IOException,
         SAXException 
  {
     final String IMPORTED_JAXP = "imports." ;
     
    /*
     * Optional:  Use an external Apache parser 
     * This is the default behavior, if setJAXPImplementation(...) is not invoked here.
     * In 'init.properties, set:
     *     DOMUtils.ImplementingClass = com.entrust.toolkit.xml.util.DOMUtilsImplApache 
     */
    // JAXParse.setJAXPImplementation(JAXParse.JAXP_DOM_APACHE /* DOM */, 
    //                                JAXParse.JAXP_SAX_APACHE /* SAX */,
    //                                JAXParse.JAXP_TRANSFORM_APACHE  /* XSLT */);
    
    /* 
     * Optional:  Alternatively, use the toolkit's internal JAXP implementation.
     * In 'init.properties, set:
     *        DOMUtils.ImplementingClass = iaik.ixsil.util.DOMUtilsImpl
     * In that case, you MUST invoke setJAXPImplementation(...) here.
     * 
     * Note: If you create the Document instance using DOMUtils.createDocumentFromXMLInstance(...)
     *       then you do NOT need to set these system properties, you only need to the
     *       set the 'DOMUtils.ImplementingClass' to use the internal iaik.ixsil.util.DOMUtilsImpl.
     */
    // JAXParse.setJAXPImplementation(IMPORTED_JAXP + JAXParse.JAXP_DOM_APACHE /* DOM */, 
    //                                IMPORTED_JAXP + JAXParse.JAXP_SAX_APACHE /* SAX */,
    //                                IMPORTED_JAXP + JAXParse.JAXP_TRANSFORM_APACHE  /* XSLT */);

    
    // Get the utility class for using JAXP
    JAXParse parser = new JAXParse();
    
    // The provided schemaLocations should at least have an URL to the XML dsig schema.
    // If you are signing or verifying XML content outside of the <Signature>, e.g. an element
    // with a particular ID, you will normally have to append additional schemas 
    // in order to parse successfully with validation.
    // Refer to the method initSignatureSchemaLocation() in the Verify.java sample 
    // for the proper format of 'schemaLocations'. 
    //
    // schemaLocations += " " + ... ;
    
    // Set the schema locations string
    parser.setSchemaLocations(schemaLocations);
    
    // Set the noNamespaceSchemaLocation for this particular document, if any,
    // as required for a successful validating parse. 
    parser.setNoNamespaceSchemaLocation(noNamespaceSchemaLocation);
    
    // Parse XML source to a Document
    return parser.parse(istream);
  }
  
  
  /**
   * Perform a validating parse of a document.
   * 
   * The provided schemaLocations should at least have the URL to the XML dsig schema.
   * If you will sign or verify XML content outside of the &lt;Signature&gt; element, e.g. an element
   * with a particular ID, amd you are providing the returned Document to IXSIL's Signer or Verifier
   * constructors, you will typically have to provide additional schemas, in order to 
   * parse successfully with validation.   Refer to the method initSignatureSchemaLocation()
   * in the Verify.java sample for the required format, which is a sequence of space-separated 
   * namespace/URL pairs.
   */
  public static Document createDOMDocument(InputStream istream, URI baseURI, String noNamespaceSchemaLocation, String schemaLocations)
  throws DOMUtilsException
  {
    Document doc = DOMUtils.createDocumentFromXMLInstance(
                        istream, 
                        baseURI, 
                        DOMUtils.VALIDATION_YES_,
                        noNamespaceSchemaLocation,
                        schemaLocations);
    return doc;
  }
  
  /**
   * Display some of the system properties.
   */
  public static void printSystemProperties()
  {
      System.out.println("Java installation directory:" + System.getProperty("java.home"));
      System.out.println("JRE version: " + System.getProperty("java.version"));
      System.out.println("Java class path:"    + System.getProperty("java.class.path"));
      System.out.println("OS:" + System.getProperty("os.name"));
      System.out.println("OS architecture:" + System.getProperty("os.arch"));
      System.out.println("OS version:" + System.getProperty("os.version"));
      System.out.println("File separator: " + System.getProperty("file.separator"));
      System.out.println("Path separator: " + System.getProperty("path.separator"));
      System.out.println("Line separator: " + iaik.ixsil.util.Utils.byteArrayToString(System.getProperty("line.separator").getBytes()));
      System.out.println("User's account name:" + System.getProperty("user.name"));
      System.out.println("User's home directory:" + System.getProperty("user.home"));
      System.out.println("User's current working directory:" + System.getProperty("user.dir"));
      System.out.println("Java VM version:" + System.getProperty("java.vm.version"));
      System.out.println("Java library path:" + System.getProperty("java.library.path"));
      System.out.println("XML DOM parser:" + System.getProperty(JAXParse.JAXP_DOM));
      System.out.println("XML SAX parser:" + System.getProperty(JAXParse.JAXP_SAX));
      System.out.println("XML XSLT implementation:" + System.getProperty(JAXParse.JAXP_TRANSFORM));
     }
     
  /**
   * Serialize a DOM document to an OutputStream.  The serialization will be
   * performed either by the <code>javax.xml.transform.Transformer</code> object 
   * from the JAXP implementation currently configured, e.g. Xalan, or by
   * an internal implementation in the Toolkit, which is based on
   * Apache Xerces 1.4 and Xalan 2.3.  This is determined by the value 
   * of the property 'DOMUtils.ImplementingClass' in the 'init.properties' file
   * that was loaded when your application initialized IXSIL.  Please refer
   * to the comments in that file.
   */
  public static void serializeDocument(Document document, OutputStream os)
  throws IOException 
  {
    try {
        DOMUtils.serializeDocument(document, os);
    }
    catch (Exception e) {
        throw new IOException("Failed to serialize the DOM document: " + e);
    }
  }    
  
  /**
   * Return a schemaLocations parameter in the format required by XML parsers:
   * The schemaLocations parameter is a space separated sequence of namespace URI's and
   * schema URL's, e.g. 
   * <p>
   * "http://www.w3.org/2000/09/xmldsig# file:/C:/test/init/schemas/Signature.xsd"
   * <p>
   * This method finds the Web Services schema URLs in the properties called
   * "schemaLocation.WsSecurity", "schemaLocation.WsUtility", and "schemaLocation.SoapEnvelope"
   * in the init properties file. </p>
   */

  public static String getWSschemaLocations()
  throws URIException
  {
    String schemaLocations = "";
     
    // In the init.properties file, you can set defaults for the SOAP, WS-Security, and 
    // WS-Utility namesspace URIs and schema locations.  They must must match the namespaces
    // in the document that you will verify.
    //
    
    String relativeSchemaLocation = IXSILInit.getInitProperty(IXSILConstants.INITPROP_SCHEMA_WS_UTILITY_);
    String schemaURL = URI.absolutizeURI(IXSILInit.getInitPropertiesURI(), new URI(relativeSchemaLocation)).toString();
    schemaLocations += " " + IXSILInit.getInitProperty(IXSILConstants.INITPROP_URI_WS_UTILITY_) + 
                                 " " + schemaURL;

    relativeSchemaLocation = IXSILInit.getInitProperty(IXSILConstants.INITPROP_SCHEMA_WS_SECURITY_);
    schemaURL = URI.absolutizeURI(IXSILInit.getInitPropertiesURI(), new URI(relativeSchemaLocation)).toString();
    schemaLocations += " " + IXSILInit.getInitProperty(IXSILConstants.INITPROP_URI_WS_SECURITY_) + 
                                 " " + schemaURL;
    
    relativeSchemaLocation = IXSILInit.getInitProperty(IXSILConstants.INITPROP_SCHEMA_SOAP_ENV_);
    schemaURL = URI.absolutizeURI(IXSILInit.getInitPropertiesURI(), new URI(relativeSchemaLocation)).toString();
    schemaLocations += " " + IXSILInit.getInitProperty(IXSILConstants.INITPROP_URI_SOAP_ENV_) + 
                                 " " + schemaURL;

    System.out.println("WS Schema locations: \""  + schemaLocations + "\"");  
    return schemaLocations ;                                   
  }
    
  
  /**
  * A helper method that constructs a schemaLocations parameter in the format
   * required by the IXSIL's Verifier class constructor.   The schemaLocations parameter
   * is a space separated sequence of namespace URI's and schema URL's, e.g. 
   *
   * "http://www.w3.org/2000/09/xmldsig# file:/C:/test/init/schemas/Signature.xsd"
   *
   * <p>
   * This method finds the signature schema URL in the property called
   * 'DOMUtils.SignatureSchema' in the init properties file. </p>
   */
  public static String getDSIGschemaLocations()
  throws URIException
  {
    String signatureSchemaLocation = "";
    
    String signatureSchema = IXSILInit.getInitProperty(IXSILConstants.INITPROP_DOMUTILS_SIGNATURE_SCHEMA_);
    URI schemaLocationURI = new URI(signatureSchema);
    String schemaURL = URI.absolutizeURI(IXSILInit.getInitPropertiesURI(), schemaLocationURI).toString();
    
    signatureSchemaLocation = IXSILConstants.NAMESPACE_URI_XMLSIGNATURE_ELEM_
                                        + " " 
                                        + schemaURL ;
    
    System.out.println("dsig schema location: \""  + signatureSchemaLocation + "\"");
    
    return signatureSchemaLocation;
  }
    
    
   /**
    * Serializes a DOM Document to a specified character encoding.
    * <p>
    * Assumes that the character encoding can encode all Unicode code points in the given XML content.
    * Assumes that the current JVM can supply the requested character encoding.
    * </p><p>
    * Note:  This method relies on an external JAXP Transformer (e.g. in Apache's xalan.jar) to perform
    * the serialization.  If the requested character encoding is not available in this particular JVM,
    * Xalan prints a warning to System.out and encodes the document into UTF-8, instead of the 
    * requested encoding.
    * </p>
    *
    * @param istream an InputStream to an XML document
    * 
    * @param ostream an OutputStream to a new XML document that will be be created.  The new document
    *                has the same XML content, but it is represented in the new character encoding.
    * 
    * @param encodingOut the character encoding to be declared in the new XML document, e.g. "UTF-8"
    */
    public static void changeCharacterEncoding( InputStream istream, OutputStream ostream, String encodingOut)
    throws IOException, SAXException, ParserConfigurationException, DOMUtilsException
    {
      // Optional:  Throw an exception if this JVM cannot supply the requested encoding.   If you do not
      // perform this check, then the Transformer implementation in Apache prints a warning to System.out
      // and encodes the document into UTF-8, instead of the requested encoding.
      //
      // Note: Actually should use the Java encoding name here (e.g. UTF16, not UTF-16), but this
      // works for most encodings.
      //
      // new OutputStreamWriter( new ByteArrayOutputStream(), encodingOut); 
        
      // Parse the source to a DOM document, without validation.
      JAXParse parser = new JAXParse(null, null);
      parser.setDocumentBuilder(false /* non validating parse */);
      Document doc = parser.parse(istream);  
      istream.close();      
      
      // serialize the DOM Document in the new character encoding.  The JAXP parser does the serialization.
      DOMUtils.serializeDocument(doc, ostream, encodingOut);
      ostream.close();
      
    }    
  
   /**
     * Converts the representation of an XML document to a different character encoding.  This is a
     * simple implementation that should work for many character encodings but does not rely on 
     * an external JAXP library.  However, if you do have JAXP support available, e.g. Apache Xerces
     * and Xalan libraries, then it is preferable to invoke
     * <code>Utils.changeCharacterEncoding(istream, ostream, encodingOut)</code> instead of this method.
     * 
     * <p>
     * Assumes that input document explicitly declares its character encoding in an XML declaration.
     * Assumes the new encoding can encode all Unicode code points in the input XML content.
     * </p>
     *
     * @param fileNameIn an XML document to be converted
     * @param fileNameOut the name of the converted XML document that will be be created
     * @param xmlEncodingOut the character encoding to use in the output file
     * 
     * @see #changeCharacterEncoding( InputStream istream, OutputStream ostream, String encodingOut)
     */
    public static void convertDocumentEncoding(String fileNameIn, String fileNameOut, String xmlEncodingOut)
    {
        if (fileNameIn.equals(fileNameOut)) {
            System.out.println("Failed to convert document because the output file name is the same as the input file name.");
            return;
        }
        
        // get index of desired XML character encoding in list of supported encodings
        int encodingIndex = -1 ;
        for (int i = 0; i < xmlEncodings.length; i++) {
            if(xmlEncodingOut.equalsIgnoreCase(xmlEncodings[i])) {
                encodingIndex = i;
                break;
            }
        }        
        
        if(encodingIndex == -1) {
            System.out.println("Failed to convert document to character encoding \"" + xmlEncodingOut + "\"");
            return;
        }
        
        String javaEncodingOut = javaEncodings[encodingIndex];  


        // Next, convert from encodingIn to encodingOut
        StringBuffer xmlDeclaration = new StringBuffer();

        try {
            int encodingInIndex = detectCharacterEncoding(fileNameIn);
            if(encodingInIndex == -1) {
                System.out.println("Failed to detect character encoding used by input document.");
                return;
            }
            
            InputStream is = new FileInputStream(fileNameIn);
            BufferedReader br = new BufferedReader(new InputStreamReader(is, javaEncodings[encodingInIndex]));

            FileOutputStream os = new FileOutputStream(fileNameOut);
            BufferedWriter bw = new BufferedWriter(new OutputStreamWriter(os, javaEncodingOut));
            
            // Some encodings need a byte order mark
            if(javaEncodingOut.equals("UnicodeBigUnmarked") || 
                javaEncodingOut.equals("UnicodeLittleUnmarked")) {
                bw.write((char) '\ufeff');
            }

            int c = -1;
            boolean readingHeader = true;
            final String ENCODING = "encoding=";            
            while ((c = br.read()) != -1) {
                if (readingHeader && (c == '>')) {
                    readingHeader = false;
                    xmlDeclaration.append((char)c);

                    // new document declares new character encoding
                    String xmlDeclarationString = xmlDeclaration.toString();
                    int start = xmlDeclarationString.indexOf(ENCODING) + ENCODING.length() + 1;
                    int end = getEncodingStringEnd(xmlDeclarationString, start);
                    xmlDeclaration.replace(start, end, xmlEncodingOut);
                    bw.write(xmlDeclaration.toString());
                    continue;
                }

                if (readingHeader) {
                    xmlDeclaration.append((char)c);
                    continue;
                }

                bw.write(c);
            }

            bw.close();
            br.close();
        } catch (IOException e) {
            System.out.println("Failure while processing XML content: " + e);
        }
    }

    /**
     * A private helper method that attempts to detect the character encoding of an XML document by reading
     * its XML declaration. This is a simple algorithm that tries to read the XML declaration using 
     * several encodings, until the declared encoding matches the encoding used to read it.
     * 
     * Note:  For a general method of autodetection of XML character encodings without external information,
     *        please refer to Appendix F.1 in "Extensible Markup Language (XML) 1.0 (Third Edition)"
     *        at <code>http://www.w3.org/TR/REC-xml</code>.
     * <p>
     * @param fileNameIn the input XML content
     *
     * @return the name of the document's character encoding from the XML declaration
     *
     * @throws IOException if the file cannot be read
     */
    private static int detectCharacterEncoding(String fileNameIn)
    throws IOException
    {
        final String ENCODING = "encoding=";
        String encoding = null;

        for (int i = 0; i < javaEncodings.length; i++) {
            
            // read to the end of the XML declaration, using a Reader with this encoding
            InputStream is = new FileInputStream(fileNameIn);
            String tryJavaEncoding = javaEncodings[i];
            BufferedReader br = new BufferedReader(new InputStreamReader(is, tryJavaEncoding));

            try {
                int c = -1;
                StringBuffer temp = new StringBuffer();                
                while ((c = br.read()) != -1) {
                
                    temp.append((char)c);                
                
                    if (c == '>') {
                        
                        br.close();
                
                        // get the document encoding declared in the XML xmlDeclaration
                        String xmlDeclarationString = temp.toString();
                        int start = xmlDeclarationString.indexOf(ENCODING) + ENCODING.length() + 1;
                        int end = getEncodingStringEnd(xmlDeclarationString, start);
                        encoding = temp.substring(start, end);
                
                        if (encoding.equalsIgnoreCase(xmlEncodings[i])) {
                            return i;
                        } else {
                            break; // try next supported encoding
                        }
                    }
                }  // while ((c = br.read()) != -1)
            }
            catch (IOException e) {
                br.close();
            }
        } // for (int i = 0; i < javaEncodings.length; i++)  

        return -1;
    }
    
  /**
   * A private helper method that returns the index of the character at the end of the 
   * character encoding name in an XML declaration, while accounting for the fact that 
   * XML allows both single quotes or double quotes around the encoding name.
   * 
   * @param xmlDeclaration is the XML declaration
   * 
   * @return the index of the character at the end the of the character encoding substring
   * 
   */
    private static int getEncodingStringEnd(String xmlDeclaration, int start)
    {
        int end = -1 ;        
        int doubleQuote = xmlDeclaration.indexOf('"', start);
        int singlequote = xmlDeclaration.indexOf('\'', start); 

        if((doubleQuote != -1) && (singlequote != -1)) {
            end = (doubleQuote < singlequote) ? doubleQuote : singlequote;
        }
        else {
            end = (doubleQuote == -1) ? singlequote : doubleQuote;                        
        }
        
        return end ;
    }  
    
    
   /**
   * A private helper that returns the URL to a provided class.
   *
   * @param c is a <code>java.lang.Class</code>
   * @return the <code>URL</code> to the resource
   */
  private static URL getUrlToResource(Class c)
  {
    final String resource = "/" + c.getName().replace('.', '/').concat(".class");
    final Class cfinal = c ;
    return (URL)java.security.AccessController.doPrivileged(
                    new java.security.PrivilegedAction() {
                        public Object run() {
                            return cfinal.getResource(resource);
                        }
                    });
                    
  }
  
}
