//===========================================================================
//
// Copyright (c)  2002-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.activeDirectory;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialCreator;
import com.entrust.toolkit.credentials.FilenameProfileWriter;
import com.entrust.toolkit.util.AuthorizationCode;
import com.entrust.toolkit.util.HttpsDirectoryClient;
import com.entrust.toolkit.util.ManagerTransport;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.LdapDirectory;

/**
 * An example of how to create an Entrust user on a PKI that uses MS Active Directory
 * as its X.509 certificate repository.  The procedure is almost identical to the
 * standard CreateCredential.java example, except that this example uses an
 * HttpsDirectoryClient object instead of a JNDIDirectory  object to communicate
 * with an SSL-enabled Active Directory proxy.
 * <p>
 * Please refer to the ActiveDirectoryServer.java sample to see how to run
 * the SSL-enabled proxy to Active Directory.
 */
public class CreateUserOnActiveDirectory
{
    public static void main(String args[])
    {
        //  Check the command-line arguments.  If they're not there, then
        // print out a usage message.
        if (args.length != 8) {
            System.out.println();
            System.out.println("usage: CreateUserOnActiveDirectory <ref no> <auth code> <profile> <initial password> <AD proxy IP address> <AD proxy port> <Authority IP address> <Authority port>");
            System.out.println();
            System.exit(0);
        }
        try {
            //  Parse in the command-line arguments.
            int index = 0;
            String refno = args[index++];
            String authcodeString = args[index++];
            String profile = args[index++];
            String password = args[index++];
            String directoryIPAddress = args[index++] ;
            String directoryProxyPort = args[index++] ;
            String managerIPAddress = args[index++];
            String authorityPort = args[index++];

            //  Display the parameters
            System.out.println();
            System.out.println("profile: " + profile);
            System.out.println("password: " + password);
            System.out.println("IP address of Entrust Authority: " + managerIPAddress);
            System.out.println("IP address of Active Directory SSL Proxy : " +
                                directoryIPAddress + ":" + directoryProxyPort);
            System.out.println();

            SecureStringBuffer refNo = new SecureStringBuffer(new StringBuffer(refno));
            AuthorizationCode authCode = new AuthorizationCode(new StringBuffer(authcodeString));
            SecureStringBuffer newPassword = new SecureStringBuffer(new StringBuffer(password));

            //  Create a user.
            CredentialCreator creator = new CredentialCreator(refNo, authCode);
            User user = new User();

            //  Connect to the infrastructure.
    	    // Just create a transport directly to the Authority in the usual way,
    	    // irrespective of whether the Directory is an Active Directory or not.
            ManagerTransport emt = new ManagerTransport(managerIPAddress, Integer.parseInt(authorityPort));

    	    // Create a transport to access Active Directory through an SSL-enabled proxy.
    	    // Note: Set the protocol here to 'http'.  The HttpsDirectoryClient class sends SSL.
    	    String directoryServerURL = "http://" + directoryIPAddress + ":" + directoryProxyPort;
    	    LdapDirectory dir = new HttpsDirectoryClient(directoryServerURL, 0);

            // Set those connections
            user.setConnections(dir, emt);
            System.out.println("Directory connection set to: " + directoryServerURL);
            System.out.println("Authority connection set to: " + managerIPAddress + ":" + authorityPort);

            //  Create a profile writer.
            FilenameProfileWriter writer = new FilenameProfileWriter(profile);
            user.setCredentialWriter(writer);

            //  Create the Entrust user profile.
            user.login(creator, newPassword);

            //  We're done.
            System.out.println("User creation successful.");
        }
        catch (Exception e) {
            System.out.println();
            System.out.println("exception thrown:");
            e.printStackTrace();
        }
    }

}
