//===========================================================================
//
// Copyright (c)  1998-2011 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.applet;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.io.ByteArrayOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.security.SecureRandom;

import javax.swing.BorderFactory;
import javax.swing.JApplet;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.ScrollPaneConstants;
import javax.swing.SwingUtilities;

import iaik.asn1.structures.AlgorithmID;

import com.entrust.toolkit.PKCS7EncodeStream;
import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.exceptions.UserBadPasswordException;
import com.entrust.toolkit.security.provider.Entrust;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.util.ManagerTransport;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.directory.JNDIDirectory;

/**
 * This class represents a Client applet that is designed to communicate 
 * securely with a corresponding Server, using PKCS#7 to secure 
 * communications.
 * 
 * <p>
 * The Client is able to encrypt and send information to a Server over a 
 * socket.  The same Entrust User is used by the Client to encrypt messages 
 * and by the Server to decrypt messages.
 * </p>
 * 
 * <p>
 * This is intended to provide an example of how the Toolkit can be used to
 * facilitate secure communication in a Client/Server environment.
 * </p>
 */
public class ClientApplet extends JApplet {

    //*****************************************************************************
    //                                  CONSTANTS
    //*****************************************************************************    

    /** The maximum block size used when receiving a message from the Server. */
    private static final int MAX_RECIEVE_BLOCKSIZE = 1024 * 16;

    /** Size of the test messages. */
    static final int[] TEST_MESSAGE_LENGTHS = new int[] { 1, 500, 2500, 5000, 25000, 75000, 750000, 3000000 };

    /** Receive connection timeout. */
    private static final int RECEIVE_CONNECTION_TIMEOUT = 5000;

    //*****************************************************************************
    //                                   FIELDS
    //*****************************************************************************    

    // Applet GUI components
    JTextArea m_progressTextArea;
    JTextField m_serverTextField;
    JTextField m_portTextField;
    JTextField m_profileTextField;
    JPasswordField m_passwordTextField;
    JButton m_connectButton;
    JButton m_testButton;
    JButton m_sendButton;
    JButton m_disconnectButton;
    JTextArea m_messageTextArea;

    // Applet Parameters   
    String m_directoryIP;
    int m_directoryPort;
    String m_managerIP;
    int m_managerPort;
    int m_serverPort;

    /** Connection to the server. */
    Socket m_socket;

    /** The User whose digital identity will be used to secure communications
     *  using PKCS7 encryption, with the server. */
    User m_user;

    /** Indicates whether or not the Applet is currently busy (only one 
     *  operation can be done at a time. */
    boolean m_busy;

    //*****************************************************************************
    //                                APPLET APIS
    //*****************************************************************************    

    /**
     * Initialize the Applet by extracting information from the Applet parameter
     * tags, prepares the Applet GUI, and initializes the Toolkit.
     */
    public void init() {
        showStatus("ClientApplet: Preparing the GUI");

        // Extract applet parameters
        m_directoryIP = getParameter("directoryIP");
        m_directoryPort = 389;
        String temp = getParameter("directoryPort");
        if (temp != null) {
            try {
                m_directoryPort = Integer.parseInt(temp);
            } catch (NumberFormatException e) {
                throw new IllegalArgumentException("Applet parameter 'directoryPort' must represent a decimal number");
            }
        }
        m_managerIP = getParameter("manangerIP");
        temp = getParameter("managerPort");
        if (temp != null) {
            try {
                m_managerPort = Integer.parseInt(temp);
            } catch (NumberFormatException e) {
                throw new IllegalArgumentException("Applet parameter 'managerPort' must represent a decimal number");
            }
        }
        String serverIP = getParameter("serverIP");
        temp = getParameter("serverPort");
        if (temp != null) {
            try {
                m_serverPort = Integer.parseInt(temp);
            } catch (NumberFormatException e) {
                throw new IllegalArgumentException("Applet parameter 'serverPort' must represent a decimal number");
            }
        }
        String serverPort = String.valueOf(m_serverPort);
        
        String profilePath = getParameter("profilePath");

        // Prepare the Applet event handler
        EventHandler eventHandler = new EventHandler();

        // General Input Components
        JPanel configurationPanel = new JPanel();
        configurationPanel.setLayout(new GridLayout(0, 1));
        
        JPanel serverInputPanel = new JPanel();
        JLabel targetLabel = new JLabel("  Server: ");
        serverInputPanel.add(targetLabel);
        m_serverTextField = new JTextField(30);
        m_serverTextField.setText(serverIP);
        m_serverTextField.addKeyListener(eventHandler);
        serverInputPanel.add(m_serverTextField);
        configurationPanel.add(serverInputPanel);
        
        JPanel portInputPanel = new JPanel();
        JLabel targetport = new JLabel("  Port: ");
        portInputPanel.add(targetport);
        m_portTextField = new JTextField(30);
        m_portTextField.setText(serverPort);
        m_portTextField.addKeyListener(eventHandler);
        portInputPanel.add(m_portTextField);
        configurationPanel.add(portInputPanel);
            
        JPanel profileInputPanel = new JPanel();
        JLabel profileLabel = new JLabel(" Profile: ");
        profileInputPanel.add(profileLabel);
        m_profileTextField = new JTextField(30);
        m_profileTextField.setText(profilePath);
        m_profileTextField.addKeyListener(eventHandler);
        profileInputPanel.add(m_profileTextField);
        configurationPanel.add(profileInputPanel);
        
        JPanel passwordInputPanel = new JPanel();
        JLabel passwordLabel = new JLabel("Password: ");
        passwordInputPanel.add(passwordLabel);
        m_passwordTextField = new JPasswordField(30);
        m_passwordTextField.setEchoChar('*');
        m_passwordTextField.addKeyListener(eventHandler);
        passwordInputPanel.add(m_passwordTextField);
        configurationPanel.add(passwordInputPanel);
        
        configurationPanel.setBorder(BorderFactory.createCompoundBorder(BorderFactory
            .createTitledBorder("Configuration Data"), BorderFactory.createEmptyBorder(5, 5, 5, 5)));

        // Control Components
        JPanel buttonPanel = new JPanel();
        m_connectButton = new JButton("Connect");
        m_connectButton.addActionListener(eventHandler);
        buttonPanel.add(m_connectButton);
        m_testButton = new JButton("Test Connection");
        m_testButton.addActionListener(eventHandler);
        buttonPanel.add(m_testButton);
        m_sendButton = new JButton("Send Text");
        m_sendButton.addActionListener(eventHandler);
        buttonPanel.add(m_sendButton);
        m_disconnectButton = new JButton("Disconnect");
        m_disconnectButton.addActionListener(eventHandler);
        buttonPanel.add(m_disconnectButton);
        buttonPanel.setBorder(BorderFactory.createTitledBorder("Controls"));

        // Message Input Components
        m_messageTextArea = new JTextArea();
        m_messageTextArea.setLineWrap(true);
        m_messageTextArea.setWrapStyleWord(true);
        JScrollPane messageScrollPane = new JScrollPane(m_messageTextArea);
        messageScrollPane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
        messageScrollPane.setBorder(BorderFactory.createCompoundBorder(BorderFactory
            .createTitledBorder("Message To Send"), BorderFactory.createEmptyBorder(5, 5, 5, 5)));

        // Main Panel
        JPanel mainPanel = new JPanel();
        mainPanel.setLayout(new BorderLayout());
        JPanel controlPanel = new JPanel();
        controlPanel.setLayout(new BorderLayout());
        controlPanel.add(configurationPanel, BorderLayout.CENTER);
        controlPanel.add(buttonPanel, BorderLayout.SOUTH);
        mainPanel.add(controlPanel, BorderLayout.WEST);
        mainPanel.add(messageScrollPane, BorderLayout.CENTER);
        mainPanel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));

        // Status Output Component
        m_progressTextArea = new JTextArea();
        m_progressTextArea.setEditable(false);
        JScrollPane progressScrollPane = new JScrollPane(m_progressTextArea);
        progressScrollPane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
        progressScrollPane.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(0, 5, 5, 5),
            BorderFactory.createCompoundBorder(BorderFactory.createTitledBorder("Progress"), BorderFactory
                .createEmptyBorder(5, 5, 5, 5))));

        // Prepare Applet
        Container applet = getContentPane();
        setContentPane(new JScrollPane(applet));
        applet.add(mainPanel, BorderLayout.NORTH);
        applet.add(progressScrollPane, BorderLayout.CENTER);
        addComponentListener(new OnShowListener());

        // Prepare for Toolkit Initialization
        printProgressMessageThreadSafe("Initializing the Entrust Toolkit... ");
        final long time = System.currentTimeMillis();

        // Initialize the Entrust Toolkit in a separate thread (allows progress 
        // messages to be posted to the GUI
        final Runnable toolkitInitializer = new Runnable() {
            public void run() {
                // Initialize the Toolkit
                m_busy = true;
                Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);
                m_user = new User();

                // Print completion message in event thread
                printProgressMessageThreadSafe("DONE [" + (System.currentTimeMillis() - time) + " ms]\n");
                printConnectionStatusThreadSafe();
                m_busy = false;
            }
        };
        new Thread(toolkitInitializer).start();
    }

    /**
     * Ensures that the client disconnects from the server if the Applet is
     * closed.
     */
    public void destroy() {
        disconnect();
    }

    //*****************************************************************************
    //                           INTERNAL HELPER METHODS
    //*****************************************************************************    

    /**
     * Connect to the server.
     */
    void connectToServer(final String profilePath, final char[] password) {
        // Prepare for connect operation
        final String serverIP = m_serverTextField.getText();
        final String port = m_portTextField.getText();
        printProgressMessageThreadSafe("\nConnecting to the Server [" + serverIP + ":" + port + "]\n");
        
        final int serverPort = Integer.parseInt(port);
        
        showStatus("ClientApplet: Busy...");
        final long startTime = System.currentTimeMillis();
        m_busy = true;

        // Execute the connect operation in a separate thread (allows progress 
        // messages to be posted to the GUI)
        final Runnable serverConnecter = new Runnable() {
            public void run() {
                try {
                    // Create PKI infrastructure connections
                    long time = 0;
                    if (m_directoryIP != null && m_managerIP != null && !m_directoryIP.equals("")
                        && !m_managerIP.equals("")) {
                        printProgressMessageThreadSafe("  Preparing PKI connections... ");
                        time = System.currentTimeMillis();
                        JNDIDirectory directory = new JNDIDirectory(m_directoryIP, m_directoryPort);
                        ManagerTransport manager = new ManagerTransport(m_managerIP, m_managerPort);
                        m_user.setConnections(directory, manager);
                        printProgressMessageThreadSafe("DONE [" + (System.currentTimeMillis() - time) + "ms]\n");
                    }

                    // Login user
                    printProgressMessageThreadSafe("  Logging in the User... ");
                    time = System.currentTimeMillis();
                    CredentialReader cr = new FilenameProfileReader(profilePath);
                    SecureStringBuffer securePassword = new SecureStringBuffer(password);
                    m_user.login(cr, securePassword);
                    securePassword.wipe();
                    printProgressMessageThreadSafe("DONE [" + (System.currentTimeMillis() - time) + "ms]\n");

                    // Establish a connection to the server
                    printProgressMessageThreadSafe("  Establishing connection with Server... ");
                    time = System.currentTimeMillis();
                    m_socket = new Socket(serverIP, serverPort);
                    m_socket.setSoTimeout(RECEIVE_CONNECTION_TIMEOUT);
                    printProgressMessageThreadSafe("DONE [" + (System.currentTimeMillis() - time) + "ms]\n");
                    printProgressMessageThreadSafe("Connection established  [" + (System.currentTimeMillis() - startTime)
                        + "ms]\n");
                } catch (FileNotFoundException e) {
                    printProgressMessageThreadSafe("FAILED\n  (!)ERROR: Profile Not Found\n");
                } catch (UserBadPasswordException e) {
                    printProgressMessageThreadSafe("FAILED\n  (!)ERROR: Incorrect Password\n");
                } catch (IOException e) {
                    printProgressMessageThreadSafe("FAILED\n  (!)ERROR: Connection Refused\n");
                    printProgressMessageThreadSafe("  (!)ERROR: " + e.toString() + "\n");
                    e.printStackTrace();
                } catch (Exception e) {
                    printProgressMessageThreadSafe("FAILED\n  (!)ERROR: An Error Occurred While Connecting\n");
                    printProgressMessageThreadSafe("  (!)ERROR: " + e.toString() + "\n");
                    e.printStackTrace();
                } finally {
                    printConnectionStatusThreadSafe();
                    m_busy = false;
                }
            }
        };
        new Thread(serverConnecter).start();
    }

    /**
     * Test the connection with the Server.
     */
    void testConnection() {
        // Prepare for connection tests
        printProgressMessageThreadSafe("\nTesting connection\n");
        showStatus("ClientApplet: Busy...");
        final long startTime = System.currentTimeMillis();
        m_busy = true;

        // Execute the connection tests in a separate thread (allows progress 
        // messages to be posted to the GUI)
        final Runnable serverConnecter = new Runnable() {
            public void run() {
                try {
                    // Test connection using various message sizes
                    SecureRandom rng = Entrust.getDefaultSecureRandomInstance();
                    for (int i = 0; i < TEST_MESSAGE_LENGTHS.length; i++) {
                        int messageLength = TEST_MESSAGE_LENGTHS[i];
                        printProgressMessageThreadSafe("  Transmitting test data [" + messageLength + " bytes]... ");
                        long time = System.currentTimeMillis();

                        // Generate the test message
                        MessageHeader header = new MessageHeader(MessageHeader.MESSAGE_TYPE_TEST, messageLength);
                        byte[] message = new byte[messageLength];
                        rng.nextBytes(message);

                        // Transmit the header and the message
                        try {
                            OutputStream os = m_socket.getOutputStream();
                            header.writeTo(os);
                            os.write(message);
                            os.flush();
                        } catch (IOException e) {
                            printProgressMessageThreadSafe("FAILED\n");
                            printProgressMessageThreadSafe("  (!)ERROR:  Error occurred while transmitting message");
                            printProgressMessageThreadSafe("  (!)ERROR:  " + e.toString() + "\n");
                            e.printStackTrace();
                            disconnect();
                            return;
                        }

                        // Receive the response
                        byte[] receivedMessage = new byte[messageLength];
                        try {
                            // Read message in reasonable size chunks
                            InputStream is = m_socket.getInputStream();
                            int bytesReceived = 0;
                            int blockSize = MAX_RECIEVE_BLOCKSIZE;
                            while (bytesReceived < messageLength) {
                                blockSize = messageLength - bytesReceived;
                                if (blockSize > MAX_RECIEVE_BLOCKSIZE)
                                    blockSize = MAX_RECIEVE_BLOCKSIZE;
                                bytesReceived += is.read(receivedMessage, bytesReceived, blockSize);
                            }
                        } catch (IOException e) {
                            printProgressMessageThreadSafe("FAILED\n");
                            printProgressMessageThreadSafe("  (!)ERROR:  Error occurred while receiving response");
                            printProgressMessageThreadSafe("  (!)ERROR: " + e.toString() + "\n");
                            e.printStackTrace();
                            disconnect();
                            return;
                        }

                        // Check response
                        int errors = 0;
                        for (int j = 0; j < messageLength; j++) {
                            if (message[j] != receivedMessage[j])
                                errors++;
                        }
                        if (errors != 0) {
                            printProgressMessageThreadSafe("FAILED\n");
                        } else {
                            printProgressMessageThreadSafe("PASSED\n");
                        }
                        printProgressMessageThreadSafe("    Transmission completed with " + errors + " byte errors ["
                            + (System.currentTimeMillis() - time) + "ms]\n");

                    }
                    printProgressMessageThreadSafe("Tests completed  [" + (System.currentTimeMillis() - startTime) + "ms]\n");
                } finally {
                    m_busy = false;
                }
                printConnectionStatusThreadSafe();
            }
        };
        new Thread(serverConnecter).start();
    }

    /**
     * Send an encrypted message to the Server.
     */
    void sendEncryptedMessage() {
        // Prepare for encrypted message transmission
        printProgressMessageThreadSafe("\nTransmitting encrypted message to Sever\n");
        showStatus("ClientApplet: Busy...");
        final long startTime = System.currentTimeMillis();
        m_busy = true;

        // Transmit encrypted message in a separate thread (allows progress 
        // messages to be posted to the GUI)
        final Runnable serverConnecter = new Runnable() {
            public void run() {

                // Execute the connection tests in a separate thread (allows progress 
                // messages to be posted to the GUI)
                try {
                    // Encrypt data
                    printProgressMessageThreadSafe("  Encrypting Data... ");
                    long time = System.currentTimeMillis();
                    ByteArrayOutputStream baos = new ByteArrayOutputStream();
                    PKCS7EncodeStream eos = new PKCS7EncodeStream(m_user, baos, PKCS7EncodeStream.SIGN_AND_ENCRYPT);
                    eos.setEncryptionAlgorithm(AlgorithmID.des_CBC);
                    eos.setDigestAlgorithm(AlgorithmID.md5);
                    eos.setBlockSize(1000);
                    byte[] data = m_messageTextArea.getText().getBytes();
                    if (data == null || data.length == 0)
                        throw new IllegalArgumentException(
                            "No data to encrypt; Please type in a message to send to the Server");
                    eos.write(data);
                    eos.close();
                    byte[] messageData = baos.toByteArray();
                    printProgressMessageThreadSafe("DONE [" + (System.currentTimeMillis() - time) + "ms]\n");

                    // Transmit encrypted message
                    printProgressMessageThreadSafe("  Transmitting Encrypted Message... ");
                    time = System.currentTimeMillis();
                    MessageHeader messageHeader = new MessageHeader(MessageHeader.MESSAGE_TYPE_REAL, messageData.length);
                    OutputStream os = m_socket.getOutputStream();
                    messageHeader.writeTo(os);
                    os.write(messageData);
                    os.flush();
                    printProgressMessageThreadSafe("DONE [" + (System.currentTimeMillis() - time) + "ms]\n");
                    printProgressMessageThreadSafe("Transmission Complete  [" + (System.currentTimeMillis() - startTime)
                        + "ms]\n");
                } catch (IllegalArgumentException e) {
                    printProgressMessageThreadSafe("FAILED\n");
                    printProgressMessageThreadSafe("  (!)ERROR: " + e.toString() + "\n");
                    e.printStackTrace();
                } catch (Exception e) {
                    printProgressMessageThreadSafe("FAILED\n");
                    printProgressMessageThreadSafe("  (!)ERROR: " + e.toString() + "\n");
                    e.printStackTrace();
                    disconnect();
                    return;
                } finally {
                    m_busy = false;
                }
                printConnectionStatusThreadSafe();
            }
        };
        new Thread(serverConnecter).start();
    }

    /**
     * Disconnect from the Server.
     */
    void disconnect() {
        if (m_socket == null) {
            printConnectionStatusThreadSafe();
            return;
        }

        // Shutdown and close the socket (ignoring errors)
        try {
            m_socket.shutdownInput();
        } catch (Exception e) {
        }
        try {
            m_socket.shutdownOutput();
        } catch (Exception e) {
        }
        try {
            m_socket.close();
        } catch (Exception e) {
        }
        m_socket = null;

        // Logout the user
        try {
            m_user.logout();
        } catch (Exception e) {
            // Ignore errors
        }
        printConnectionStatusThreadSafe();
    }

    //*****************************************************************************
    //                           STATUS MESSAGE APIS
    //*****************************************************************************

    /**
     * Prints a progress message to the 'progress' text area indicating the status
     * of the connection; Prints the Applet status as ready.
     * 
     * <p>
     * Ensures that the Applet components are modified in the event-dispatching
     * thread; text is updated in real-time.
     * </p>
     */
    void printConnectionStatusThreadSafe() {
        if (SwingUtilities.isEventDispatchThread()) {
            printConnectionStatus();
        } else {
            final Runnable printProgress = new Runnable() {
                public void run() {
                    printConnectionStatus();
                }
            };
            SwingUtilities.invokeLater(printProgress);
        }
    }

    /**
     * Prints a progress message to the 'progress' text area indicating the status
     * of the connection; Prints the Applet status as ready.
     */
    void printConnectionStatus() {
        if (m_socket == null) {
            printProgressMessage("\n[Status: Not connected (ready to connect)]\n");
        } else {
            printProgressMessage("\n[Status: Connected (ready to transmit)]\n");
        }
        showStatus("ClientApplet: Ready");
    }

    /**
     * Prints a progress message to the 'progress' text area..
     * 
     * <p>
     * Ensures that the Applet components are modified in the event-dispatching
     * thread; text is updated in real-time.
     * </p>
     * 
     * @param message
     *      the progress message
     */
    void printProgressMessageThreadSafe(final String message) {
        if (SwingUtilities.isEventDispatchThread()) {
            printProgressMessage(message);
        } else {
            final Runnable printProgress = new Runnable() {
                public void run() {
                    printProgressMessage(message);
                }
            };
            SwingUtilities.invokeLater(printProgress);
        }
    }

    /**
     * Prints a progress message to the 'progress' text area.
     * 
     * @param message
     *      the progress message
     */
    void printProgressMessage(final String message) {
        m_progressTextArea.append(message);
        // Make sure the last line is always visible
        m_progressTextArea.setCaretPosition(m_progressTextArea.getDocument().getLength());
    }

    //*****************************************************************************
    //                           INTERNAL EVENT HANDLERS
    //*****************************************************************************

    /**
     * This class is handles the events that occur within the Applet; specifically
     * button presses and pressing 'Enter' in a text field.
     */
    class EventHandler implements ActionListener, KeyListener {

        /**
         * Detects when a button in the Applet GUI is pressed and takes the 
         * appropriate action.
         */
        public void actionPerformed(ActionEvent actionEvent) {
            // Only process events when not busy
            if (m_busy)
                return;

            // Determine which button was pressed
            Object eventSource = actionEvent.getSource();
            if (eventSource == m_connectButton) {
                if (m_socket != null) {
                    printProgressMessage("\n-- Already Connected\n");
                } else {
                    connectToServer(m_profileTextField.getText(), m_passwordTextField.getPassword());
                }
            } else if (eventSource == m_testButton) {
                if (m_socket != null) {
                    testConnection();
                } else {
                    printProgressMessage("\n-- Not Connected\n");
                }
            } else if (eventSource == m_sendButton) {
                if (m_socket != null) {
                    sendEncryptedMessage();
                } else {
                    printProgressMessage("\n-- Not Connected\n");
                }
            } else if (eventSource == m_disconnectButton) {
                if (m_socket != null) {
                    disconnect();
                } else {
                    printProgressMessage("\n-- Not Connected\n");
                }
            }
        }

        /**
         * Ignore all 'keyPressed' events.
         */
        public void keyPressed(KeyEvent keyEvent) {
        }

        /**
         * Ignore all 'keyTyped' events.
         */
        public void keyTyped(KeyEvent keyEvent) {
        }

        /**
         * When the 'Enter' key is released within any of the text fields
         * either transfer to an appropriate text field, or connect to server
         * if appropriate and send an encrypted message.
         */
        public void keyReleased(KeyEvent keyEvent) {
            // Only recognize release of the 'Enter' key
            if (keyEvent.getKeyCode() != KeyEvent.VK_ENTER)
                return;
            // Only process events when not busy
            if (m_busy)
                return;

            Object eventSource = keyEvent.getSource();
            if (eventSource == m_passwordTextField) {
                // Connect to the server
                if (m_socket == null)
                    connectToServer(m_profileTextField.getText(), m_passwordTextField.getPassword());
            } else if (eventSource == m_serverTextField) {
                // Transfer focus to profile text field
                m_profileTextField.requestFocus();
            } else if (eventSource == m_profileTextField) {
                // Transfer focus to password text field and select password text
                m_passwordTextField.requestFocus();
                m_passwordTextField.selectAll();
            }
        }
    }
    
    /**
     * This class detects when the Applet is shown and displays a status
     * message.
     */
    class OnShowListener extends ComponentAdapter {
        public void componentShown(ComponentEvent componentEvent) {
            showStatus("ClientApplet: Busy...");
        }
    }
}