//===========================================================================
//
// Copyright (c)  2003-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.archive;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.util.LinkedList;
import java.util.List;

import iaik.x509.X509Certificate;

import com.entrust.toolkit.KeyAndCertificateSource;
import com.entrust.toolkit.User;
import com.entrust.toolkit.archive.ArchiveCreator;
import com.entrust.toolkit.archive.ArchiveInputStream;
import com.entrust.toolkit.archive.ArchiveResults;
import com.entrust.toolkit.archive.PemOptions;
import com.entrust.toolkit.asn1.structures.AlgorithmID;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.exceptions.UserNotLoggedInException;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.util.SecureStringBuffer;

/**
 * Example to demonstrate the usage of the Archive classes.
 * <p>The example performs the following major steps:</p>
 * <ol>
 * <li>Log in to a supplied Entrust user.</li>
 * <li>Creates an Entrust Archive by encrypting and signing a supplied file</li>
 * <li>Displays the Archive</li>
 * <li>Creates an ArchiveInputStream to read the Archive</li>
 * <li>Reads all the data from the Archive</li>
 * <li>Displays the results of signature and certificate verification</li>
 * </ol>
 * <p>This sample uses the <code>ArchiveCreator</code> class, which is 
 * not recommended as the Entrust Archive (.ent) format is a legacy, non-interoperable
 * format. PKCS7 and SMIME are preferred. It is used as a convenience for
 * people who do not have any .ent files.</p>
 *
 * <p>Usage:</p>
 * <pre>ArchiveExample path_to_epf password infile [{cert files}]
 * </pre>
 * <dl>
 * <dt>path_to_epf</dt><dd>Path to EPF file. For example, data/userdata/RSAUser1.epf</dd>
 * <dt>password</dt><dd>The password for the EPF. For example, ~Sample7~</dd>
 * <dt>infile</dt><dd>The file to protect. For example, data/testfiles/test.txt</dd>
 * <dt>cert_files</dt><dd>Optional list of files containing certificates to encrypt for. For example, data/userdata/RSAUser2Encryption.cer</dd>
 * </dl>
 * <p>
 * For example, from the examples directory, 
 * <pre>
 * javac -sourcepath source -d classes -classpath ../lib/enttoolkit.jar source/com/entrust/toolkit/examples/archive/ArchiveExample.java
 * java -classpath classes;../lib/enttoolkit.jar com.entrust.toolkit.examples.archive.ArchiveExample
 *   data/userdata/RSAUser1.epf ~Sample7~ data/testfiles/test.txt data/userdata/RSAUser2Encryption.cer
 * </pre>
 * </p>
 */
public class ArchiveExample
{
    /**
     * Main program for ArchiveExample.
     * @param args
     *     Program arguments. See the class information for details.
     */
    public static void main(String args[])
    {
        User user = new User();

        System.out.println("ArchiveExample: shows simple usage of the Archive classes");
        //  Check the command-line arguments.  If they're not there, then
        // print out a usage message.
        if (args.length < 3)
        {
            System.out.println();
            System.out.println("usage: ArchiveExample <profile> <password> <input file> {<cert files>}");
            System.out.println();
            return;
        }
        //  Parse in the command-line arguments.
        int index = 0;
        String epfFile = args[index++];
        String password = args[index++];
        File inputFile = new File(args[index++]);

        //  Display the parameters
        System.out.println();
        System.out.println("profile: " + epfFile);
        System.out.println("password: " + password);
        System.out.println("input file: " + inputFile.getAbsolutePath());
        System.out.println();

        try
        {
            // Initialize the Toolkit.
            Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);

            // Check if extra recipients were specified, and
            // read all of their certificates in if so.
            // Do this now so that we don't waste time logging in
            // any of the recipient certificates cannot be read.
            List extraCertificates = new LinkedList();
            while (index < args.length)
            {
                System.out.println("adding recipient: " + args[index]);
                X509Certificate extraCert = new X509Certificate(new FileInputStream(args[index++]));
                extraCertificates.add(extraCert);
            }

            // Log into an Entrust user, whose credentials we will use
            // to create an Entrust Archive, and then to verify it.
            CredentialReader credentialReader = new FilenameProfileReader(epfFile);
            System.out.println("Logging in");
            user.login(credentialReader, new SecureStringBuffer(new StringBuffer(password)));
            System.out.println("Login complete");

            // Set the options to be used when creating the Archive.
            // We will compress, encrypt with 128 bit AES, sign, and Base64 encode
            // the output.
            PemOptions createOptions = new PemOptions();
            createOptions.setBase64Encoded(true);
            createOptions.setEncrypted(true);
            createOptions.setSigned(true);
            createOptions.setCompressed(true);
            createOptions.setEncryptionAlgorithm(AlgorithmID.CipherAlgs.id_aes128_CBC, 128);

            // Create a stream to which the archive will be written.
            // The adventurous person could send this to a FileOutputStream,
            // then decrypt the results with Entrust Entelligence.
            ByteArrayOutputStream dataOutStream = new ByteArrayOutputStream();

            // Create the object that will create the Archive.
            ArchiveCreator archiveCreator = new ArchiveCreator(user, inputFile, createOptions, dataOutStream);

            // Add any extra recipients. If the recipients come from a different CA
            // for which there is no explicit cross-certification, the addTrustedRecipients
            // method can be used instead.
            if (extraCertificates.size() > 0)
            {
                archiveCreator.addRecipients(
                    (X509Certificate[]) extraCertificates.toArray(
                        new X509Certificate[extraCertificates.size()]));
            }

            // Now go about the business of creating the archive.
            System.out.println("Creating Entrust Archive");
            while (archiveCreator.process() != -1)
            {
                // Could use this opportunity to provide some feedback
                // about what's going on.
            }

            // Display the Archive. This will only work if it is Base64 encoded,
            // otherwise it will display binary data.
            System.out.println("The contents of the Archive are:");
            System.out.println(dataOutStream.toString());
            System.out.println();

            // The archive has been created. Now it's time to verify it.
            ByteArrayInputStream dataInStream = new ByteArrayInputStream(dataOutStream.toByteArray());
            KeyAndCertificateSource keyAndCertificateSource = new KeyAndCertificateSource(user);
            ArchiveInputStream archiveInputStream =
                new ArchiveInputStream(keyAndCertificateSource, dataInStream);

            // Before reading data from the Archive, see what the included file name was. 
            String includedFileName = archiveInputStream.getIncludedFileName();
            System.out.println("File name in the archive is: " + includedFileName);
            // At this time, a FileOutputStream could be set up to write the included
            // file back to disk. Caution is required, as the file name included in an
            // Entrust Archive may be either absolute or relative, and the directory
            // separators may be either Windows-style '\' or UNIX-style '/'. The String
            // returned by getIncludedFileName will be exactly what is in the Archive,
            // no attempt is made to make it suitable for the current environment.

            // Read the data from the archive.
            System.out.println("Reading the included file from the Archive");
            byte[] archiveData = new byte[1024];
            int archiveBytesRead = archiveInputStream.read(archiveData);
            while (archiveBytesRead != -1)
            {
                // A real application would do something useful to the bytes read
                archiveBytesRead = archiveInputStream.read(archiveData);
            }
            System.out.println();
            System.out.println("Finished reading from archive");

            // Close the stream, causing signatures to be validated.
            archiveInputStream.close();

            // This step could be done before or after closing the stream.
            PemOptions archiveOptions = archiveInputStream.getFileProtectionOptions();
            System.out.println("Archive was protected with the following options:");
            System.out.println("Encrypted: " + archiveOptions.isEncrypted());
            if (archiveOptions.isEncrypted())
            {
                System.out.println(
                    "Encryption algorithm: " + archiveOptions.getEncryptionAlgorithm().getJCAAlgName());
                System.out.println("Encryption key length: " + archiveOptions.getKeyLength());
            }
            System.out.println("Signed: " + archiveOptions.isSigned());
            System.out.println("Compressed: " + archiveOptions.isCompressed());
            System.out.println("Base64 encoded: " + archiveOptions.isBase64Encoded());

            // See what the results of reading the archive were.
            // This merely displays the results to the screen,
            // in reality some trust decision should be made based on these
            // results.
            ArchiveResults archiveResults = archiveInputStream.getArchiveResults();
            if (archiveResults.isCrcOk())
            {
                System.out.println("The Archive CRC is OK");
            }
            else
            {
                System.err.println("The Archive CRC is NOT okay, Archive header was tampered with");
            }

            if (!archiveResults.isSignatureValid())
            {
                if( archiveResults.getSignatureException() == null )
                {
                    System.err.println(
                        "The signature on the Archive is NOT okay, the included data has been tampered with.");
                }
                else
                {
                    System.err.println(
                        "The signature on the Archive is NOT okay, the signature has been tampered with.");
                }
            }
            else
            {
                System.out.println("The signature on the Archive is OK");
            }

            if (archiveResults.getEncryptionCertificateException() == null)
            {
                System.out.println("The Archive creator's encryption certificate is OK");
            }
            else
            {
                System.err.println("The Archive creator's encryption certificate FAILED to validate");
            }

            if (archiveResults.getVerificationCertificateException() == null)
            {
                System.out.println("The Archive creator's verification certificate is OK");
            }
            else
            {
                System.err.println("The Archive creator's verification certificate FAILED to validate");
            }
        }
        catch (Exception e)
        {
            System.out.println();
            System.out.println("exception thrown:");
            e.printStackTrace();
        }
        finally
        {
            // Ensure that the user is logged out.
            if (user.isLoggedIn())
            {
                try
                {
                    user.logout();
                }
                catch (UserNotLoggedInException e)
                {
                }
            }
        }
    }
}
