//===========================================================================
//
// Copyright (c)  2002-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.cache;

import com.entrust.toolkit.*;
import com.entrust.toolkit.credentials.*;
import com.entrust.toolkit.x509.directory.*;
import com.entrust.toolkit.util.*;
import com.entrust.toolkit.x509.revocation.CachedCRLRS;
import com.entrust.toolkit.x509.revocation.ArchiveCRLCache;

/**
 * This class demonstrates how to retrieve user's cached CRLs after an
 * online login.
 * <p>
 * Usage:
 * <pre>
 * CacheCRL path_to_epf password directory_ip_address
 * </pre>
 * <dl>
 * <dt>path_to_epf</dt><dd>Path to EPF file. For example, data/userdata/RSAUser1.epf</dd>
 * <dt>password</dt><dd>The password for the EPF. For example, ~Sample7~</dd>
 * <dt>directory_ip_address</dt><dd>The IP address of the Directory to connect to.
 *    For example, 127.0.0.1</dd>
 * </dl>
 * 
 */
public class CacheCRL
{
    /**
     * The main driver for the program.
     * 
     * @param args
     *     the program arguments. See the class documentation for program usage.
     */
    public static void main(String[] args)  {
        if ( args.length != 3 )
        {
            System.out.println("Usage: java CacheCRL <epf> <password> <directory ip>");
            return;
        }
        
        new CacheCRL(args[0], args[1], args[2]);
    }
    
    public CacheCRL(String profile, String password, String dirIP)
    {
        try {
            User user = new User();
         
            // set the connections
            JNDIDirectory dir = new JNDIDirectory (dirIP,389);
            user.setConnections(dir, null);
         
            // login
            CredentialReader cr = new FilenameProfileReader(profile);
            System.out.println("Logging in to " + profile );
            user.login(cr, new SecureStringBuffer(new StringBuffer(password)));
           
            System.out.println("Login status:");
            System.out.println(user.getStatus());
            
            // Do a certificate validation to force something in to the cache. With
            // a 7.0 Entrust PKI a user may not do any certificate validation in
            // the login call.
            user.validate(user.getVerificationCertificate());
            
            // display the cached CRLs. 
            System.out.println();
            System.out.println("==========Cached CRLs===============");
            CachedCRLRS cachedCRLRS = user.getCertVerifier().getRevocationStore().getMemoryCRLCache();
            ArchiveCRLCache archiveCRLCache = new ArchiveCRLCache(user.getCertVerifier() );
            archiveCRLCache.addMemoryCache( cachedCRLRS );
            // You can write them to another destination
            // by changing the output stream of the write method
            archiveCRLCache.write( System.out, ArchiveCRLCache.CRL_AND_ARL );
             
            user.logout();
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }
}