//===========================================================================
//
// Copyright (c)  2003-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.capi;

import java.io.File;
import java.util.Iterator;
import java.util.List;

import com.entrust.toolkit.JniInitializer;
import com.entrust.toolkit.capi.CapiConstants;
import com.entrust.toolkit.capi.CapiHandle;
import com.entrust.toolkit.capi.CryptProvider;
import com.entrust.toolkit.capi.ProviderType;
import com.entrust.toolkit.security.provider.Initializer;

/**
 * Sample to list all key containers belonging to a Cryptographic Service
 * Provider.
 * <p>
 * The name of the CSP can be specified on the command line. If no
 * arguments are provided, then containers are listed for the
 * Microsoft Strong CSP.
 * </p>
 * <p>
 * The names of all CSPs can be found by running the <code>EnumProviders</code>
 * sample.
 * </p>
 * <p>
 * To run, from the examples directory:
 * <pre>
 * java -classpath ..\lib\enttoolkit.jar;classes com.entrust.toolkit.examples.capi.EnumKeyContainers "Entrust Enhanced Cryptographic Provider"
 * </pre>
 * Would list all containers for the Entrust CSP.
 * </p>
 * <p>
 * Note that this sample requires the native library JniCapi.dll to be available.
 * </p>
 * 
 * @see EnumProviders
 */
public class EnumKeyContainers
{
    /**
     * The main driver for the program.
     * 
     * @param args
     *     Program arguments. If present, <code>args[0]</code> is used
     * as the CSP name. All other entries are ignored. 
     */
    public static void main(String[] args)
    {
        // These lines load the native library that contains the native CAPI
        // calls relative to the current working directory. The path could be
        // changed to any absolute path, or the lines can be removed completely if
        // JniCapi.dll is installed in the system path.
        File nativeLibrary = new File("../lib/win32/JniCapi.dll");
        JniInitializer.initializeCapi(nativeLibrary.getAbsolutePath());

        // Insert the Entrust Provider in to the list of Providers so that
        // crypto operations will work.
        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);

        // If arguments were supplied, use the first one as the name of the
        // to list key containers for.
        String providerName;
        if (args.length > 0)
        {
            providerName = args[0];
        }
        else
        {
            providerName = CapiConstants.MS_STRONG_PROV;
        }

        CryptProvider provider = null;
        try
        {
            ProviderType providerType = ProviderType.PROV_RSA_FULL;
            System.out.println("Key containers for provider \'" + providerName + "\'");
            // Get a handle to the CSP. Given that no cryptographic capabilities
            // are required, use the CRYPT_VERIFYCONTEXT flag.
            provider =
                CryptProvider.getInstance(
                    null,
                    providerName,
                    providerType,
                    CapiConstants.CRYPT_VERIFYCONTEXT);

            // Get the list of all containers, and loop through them,
            // displaying the name of each container.
            int index = 1;
            List containers = provider.getContainers();
            Iterator containerIt = containers.iterator();
            while (containerIt.hasNext())
            {
                String containerName = (String) containerIt.next();
                System.out.println(index + ": " + containerName);
                ++index;
            }
            if (1 == index)
            {
                System.out.println("CSP has no key containers");
            }
        }
        catch (Exception e)
        {
            System.out.println("Caught exception: ");
            e.printStackTrace();
        }
        finally
        {
            // Make sure the provider is always closed
            if (provider != null)
            {
                provider.close();
            }
        }

        // Display debugging information about the currently allocated
        // native CAPI objects.  It should display zeroes for all values.
        CapiHandle.debugInfo(true);
    }
}
