//===========================================================================
//
// Copyright (c)  2003-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.capi;

import java.io.File;
import java.util.Iterator;
import java.util.List;

import com.entrust.toolkit.JniInitializer;
import com.entrust.toolkit.capi.CapiConstants;
import com.entrust.toolkit.capi.CapiHandle;
import com.entrust.toolkit.capi.CryptProvider;
import com.entrust.toolkit.capi.ProviderAlgorithmInfo;
import com.entrust.toolkit.capi.ProviderInfo;
import com.entrust.toolkit.capi.ProviderType;

/**
 * Sample to list all CAPI provider types, all CAPI providers, and algorithms
 * supported by the default provider.
 * <p>
 * To run, from the examples directory:
 * <pre>
 * java -classpath ..\lib\enttoolkit.jar;classes com.entrust.toolkit.examples.capi.EnumProviders
 * </pre>
 * </p>
 * <p>
 * Note that this sample requires the native library JniCapi.dll to be available.
 * </p>
 */
public class EnumProviders
{
    /**
     * The main program. Displays information about installed providers,
     * provider types, and supported algorithms.
     * 
     * @param args
     *     not used.
     */
    public static void main(String[] args)
    {
        // These lines load the native library that contains the native CAPI
        // calls relative to the current working directory. The path could be
        // changed to any absolute path, or the lines can be removed completely if
        // JniCapi.dll is installed in the system path.
        File nativeLibrary = new File("../lib/win32/JniCapi.dll");
        JniInitializer.initializeCapi(nativeLibrary.getAbsolutePath());
        
        try
        {
            // Get the list of installed CAPI provider types, then iterate
            // over the list displaying information about the provider type.
            System.out.println("Available Provider Types.");
            System.out.println("__________________________________________________");
            int index = 0;
            ProviderInfo info;
            List providerTypes = CryptProvider.getProviderTypes();
            Iterator iterator = providerTypes.iterator();
            while (iterator.hasNext())
            {
                info = (ProviderInfo)iterator.next();
                System.out.println(
                        index
                        + ". Type = "
                        + info.getProviderType()
                        + ", type name = "
                        + info.getProviderName());
                ++index;
            }

            // Get the list of installed CAPI providers, then iterate
            // over the list displaying information about each provider.
            System.out.println();
            System.out.println("Available Providers.");
            System.out.println("__________________________________________________");
            index = 0;
            List providers = CryptProvider.getProviders();
            iterator = providers.iterator();
            while (iterator.hasNext())
            {
                info = (ProviderInfo)iterator.next();
                System.out.println(
                    index
                        + ". Type = "
                        + info.getProviderType()
                        + ", provider name = "
                        + info.getProviderName());
                ++index;
            }
            System.out.println("\r\n");


            // Now acquire a handle to the default RSA provider, and list the
            // algorithms it supports.
            CryptProvider provider = CryptProvider.getInstance(
                    null,
                    null,
                    ProviderType.PROV_RSA_FULL,
                    CapiConstants.CRYPT_VERIFYCONTEXT);

            if (provider == null)
            {
                System.out.println("Error getting handle to provider");
                return;
            }

            System.out.println("Successfully acquired handle to default RSA provider");
            System.out.println("Provider name:  " + provider.getProviderInfo().getProviderName());

            System.out.println();
            System.out.println("Algorithms supported by default provider\n");
            System.out.println("     Algid      Bits      Type        Name         Algorithm");
            System.out.println("____________________________________________________________");

            index = 0;
            List algs = provider.getSupportedAlgorithms();
            Iterator it = algs.iterator();
            while (it.hasNext())
            {
                // A ProviderAlgorithmInfo is very much like a PROV_ENUMALGS_EX structure
                // in CAPI. It stores useful information about each algorithm
                ProviderAlgorithmInfo algInfo = (ProviderAlgorithmInfo) it.next();
                System.out.println(
                index
                        + ": "
                        + algInfo.getAlgId()
                        + " bits: "
                        + algInfo.getDefaultLen()
                        + " type: "
                        + algInfo.getAlgorithmType()
                        + " name: "
                        + algInfo.getName()
                        + " full name: "
                        + algInfo.getLongName());
                ++index;
            }

            // Make sure the provider handle is closed to prevent memory or
            // resource leaks.
            provider.close();

        }
        catch (Exception e)
        {
            System.out.println("Caught exception: ");
            e.printStackTrace();
        }
        
        // Display debugging information about the currently allocated
        // native CAPI objects.  It should display zeroes for all values.
        CapiHandle.debugInfo(true);

    }
}
