//===========================================================================
//
// Copyright (c)  2003-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.crypto;

import java.io.FileInputStream;
import java.io.FileOutputStream;

import javax.crypto.Cipher;
import javax.crypto.CipherInputStream;
import javax.crypto.CipherOutputStream;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;

import com.entrust.toolkit.security.provider.Initializer;

/**
 * This example demonstrates how to use a CipherInputStream and 
 * CipherOutputStream in conjunction with an Entrust JCA Cipher algorithm.
 * 
 * <p>
 * It shows how to create an encrypted file using a CipherOutputStream, and
 * then how to decrypt that file using a CipherInputStream.
 * </p>
 */
public class CipherStreamExample {

    /**
     * The main program.
     * <p></p>
     * 
     * @param args
     *      program arguments
     */
    public static void main(String args[]) {
        // Ensure that the arguments are valid
        if (args.length != 2) {
            System.out.println("Usage: CipherStreamExample <file to encrypt> <encrypted file>");
            return;
        }
        String plaintextFile = args[0];
        String ciphertextFile = args[1];

        // Ensure that Entrust's providers have been installed
        System.out.print("Installing providers... ");
        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);
        System.out.println("DONE");
        System.out.println();

        try {
            // Request an instance of the DESede cipher and key generation 
            // algorithms from the Entrust provider.  The requested DESede cipher
            // will operate in ECB mode, and use PKCS5 padding.
            Cipher cipher = Cipher.getInstance("DESede/ECB/PKCS5Padding", "Entrust");
            KeyGenerator keyGen = KeyGenerator.getInstance("DESede", "Entrust");

            // Generate a key that can be used with the DESede cipher
            System.out.print("Generating key... ");
            keyGen.init(192);
            SecretKey key = keyGen.generateKey();
            System.out.println("DONE");
            System.out.println();

            // Create a CipherOutputStream, attached to a FileOutputStream of the
            // plaintext file, that is initialized for encryption using the key 
            // that was generated 
            cipher.init(Cipher.ENCRYPT_MODE, key);
            FileOutputStream encryptedOutputStream = new FileOutputStream(ciphertextFile);
            CipherOutputStream cos = new CipherOutputStream(encryptedOutputStream, cipher);

            // Encrypt the file by reading all the data it contains, and writing
            // that data to the CipherOutputStream
            System.out.print("Encrypting the plaintext file... ");
            FileInputStream is = new FileInputStream(plaintextFile);
            byte[] buffer = new byte[1024];
            int i = 0;
            while ((i = is.read(buffer)) != -1) {
                cos.write(buffer, 0, i);
            }
            cos.close();
            System.out.println("DONE");
            System.out.println();

            // Create a CipherInputStream, attached to a FileInputStream of the
            // plaintext file, that is initialized for decryption using the key 
            // that was generated 
            cipher.init(Cipher.DECRYPT_MODE, key);
            FileInputStream encryptedInputStream = new FileInputStream(ciphertextFile);
            CipherInputStream cis = new CipherInputStream(encryptedInputStream, cipher);

            // Decrypt the file by reading all the data it contains, and writing
            // that data to the FileOutputStream
            System.out.print("Decrypting the ciphertext file... ");
            FileOutputStream fos = new FileOutputStream(plaintextFile + ".decrypted.txt");
            i = 0;
            while ((i = cis.read(buffer)) != -1) {
                fos.write(buffer, 0, i);
            }
            fos.close();
            System.out.println("DONE");
            System.out.println();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}