//===========================================================================
//
// Copyright (c)  2003-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.crypto;

import java.security.Provider;
import java.security.Security;
import java.util.Iterator;
import java.util.Map;

import com.entrust.toolkit.security.provider.Initializer;

/**
 * This example demonstrates how to determine the various cryptographic algorithms
 * available through the Java Cryptography Architecture (JCA).
 * 
 * <p>
 * The JCA refers to a framework for accessing and developing cryptographic 
 * functionality for the Java platform.  For further information on how the JCA
 * is used, please refer to Java's 
 * <a href=http://download.oracle.com/javase/1.5.0/docs/guide/security/index.html>
 * Security Documentation</a> documentation.
 * </p>
 */
public class JcaAlgorithmImplementations {

    /**
     * The main program.
     * <p></p>
     * 
     * @param args
     *      not used
     */
    public static void main(String[] args) {

        // Ensure that Entrust's providers have been installed
        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);

        // Print the installed JCA algorithms
        printInstalledAlgorithms("Cipher");
        printInstalledAlgorithms("KeyAgreement");
        printInstalledAlgorithms("KeyGenerator");
        printInstalledAlgorithms("KeyPairGenerator");
        printInstalledAlgorithms("Mac");
        printInstalledAlgorithms("MessageDigest");
        printInstalledAlgorithms("SecureRandom");
        printInstalledAlgorithms("Signature");
    }

    /**
     * Prints the list of algorithms provided by each installed JCA provider
     * for a particular algorithm type.
     * <p></p>
     * 
     * @param algorithmType
     *      the type of JCA algorithm (Cipher, Mac, ...)
     */
    public static void printInstalledAlgorithms(String algorithmType) {
        System.out.println("Installed JCA '" + algorithmType + "' algorithms\n");

        // Look for the requested algorithms in each installed JCA provider
        Provider[] providers = Security.getProviders();
        int size = providers.length;
        for (int i = 0; i < size; i++) {
            Provider provider = providers[i];
            System.out.println(provider.getName());

            // These will contain a list of the algorithms and the algorithm 
            // aliases that have been found
            StringBuffer algorithms = new StringBuffer("[");
            StringBuffer aliases = new StringBuffer("[");

            // Look at each entry the provider has
            Iterator iterator = provider.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry entry = (Map.Entry) iterator.next();
                String key = (String) entry.getKey();

                // Ignore entries that are not for the requested algorithm type
                // ex: Signature.SHA1withDSA=sun.security.provider.DSA (when not requesting a Signature)
                int index = key.indexOf(algorithmType);
                if (index < 0)
                    continue;

                // Ignore entries that DO NOT refer to algorithms, but instead 
                // provide addition information about another entry 
                // ex: Signature.SHA1withDSA ImplementedIn=Software
                // ex: Signature.SHA1withDSA KeySize=1024
                if (key.indexOf(' ') >= 0)
                    continue;

                if (index == 0) {
                    // We have an algorithm entry, so added it to the list of 
                    // algorithms that have been found
                    // ex: Signature.SHA1withDSA=sun.security.provider.DSA (when requesting a Signature)
                    algorithms.append(key.substring(key.indexOf('.', index) + 1) + ", ");
                } else {
                    // We have an algorithm alias entry, so added it to the list 
                    // of algorithm aliases that have been found
                    // ex: Alg.Alias.Signature.SHA-1/DSA=SHA1withDSA (when requesting a Signature)
                    aliases.append(key.substring(key.indexOf('.', index) + 1) + ", ");
                }
            }

            // Formatting and printing of the algorithm and algorithm alias lists
            System.out.print("  algorithms: ");
            if (algorithms.length() == 1) {
                System.out.println("none");
            } else {
                algorithms.deleteCharAt(algorithms.length() - 1);
                algorithms.deleteCharAt(algorithms.length() - 1);
                algorithms.append(']');
                System.out.println(algorithms.toString());
            }
            System.out.print("  aliases: ");
            if (aliases.length() == 1) {
                System.out.println("none");
            } else {
                aliases.deleteCharAt(aliases.length() - 1);
                aliases.deleteCharAt(aliases.length() - 1);
                aliases.append(']');
                System.out.println(aliases.toString());
            }
        }
        System.out.println();
    }
}