//===========================================================================
//
// Copyright (c)  2000-2012 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.jsse;

import java.io.BufferedWriter;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.security.KeyStore;
import java.security.NoSuchAlgorithmException;
import java.security.Security;
import java.util.ArrayList;
import java.util.List;

import javax.net.ssl.*;

import com.entrust.toolkit.security.provider.Entrust;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.jsse.provider.EntrustJSSEProvider;

/**
 * A simple example that shows how you can use the Toolkit to do SSL within the
 * JSSE API.  This is the client.  You can use the {@link JSSEServer} sample
 * as the server.
 */
public class JSSEClient {
    
    // Initialize security providers and debug trace
    //
    static {

        // ALWAYS register the Toolkit's JCE providers.
        // Since this sample is demonstrating JSSE, initialize the Entrust JSSE Provider as well.
        Initializer.getInstance().setProviders(Initializer.MODE_ENTRUST_FIRST);
        Security.addProvider(new EntrustJSSEProvider());
        
        // Optional: Only when debugging, enable the trace from Sun's "SunJSSE"
        // provider.  Warning: Some secret data will be written to System.out !
        //System.setProperty("javax.net.debug", "ssl");
    }    
    
    public static void main(String[] args) {
        
        //  Get command-line arguments.
        if(args.length < 4 || args.length > 5) {
            System.out.println("Usage: java JSSEClient <EPF Name> <Password> <Host Address> <Port> [<IncludeSNI>]");
            System.out.println("   where <EPF Name> is the profile that will be used as the Java TrustStore for the TLS connection");
            System.out.println("   where <Password> is the password to the EPF");
            System.out.println("   where <Host Address> is the hostname or IP address of the remote server");
            System.out.println("   where <Port> is the port that the remote server is listening on");
            System.out.println("   where <IncludeSNI> is a boolean indicating whether to include the Server Name Indication (SNI)\n" +
                               "      extension in the TLS Client Hello. The default is 'false'.  If set to 'true', then the <Host Address>\n" +
                               "      should specify the fully qualified domain name (FQDN) of the remote (virtual) server, which should match\n" +
                               "      the CN entry in the Subject DN (or Subject Alternative Name) of the server certificate.");
            return;
        }

        String epf = args[0];
        String pw = args[1];
        String host = args[2];
        int port = Integer.parseInt(args[3]);

        boolean includeSNI = false;
        if (args.length == 5) {
            includeSNI = Boolean.valueOf(args[4]);
        }

        try {
            SSLContext context;
            try {
                // Attempt to create a context that supports TLS 1.2
                context = SSLContext.getInstance("TLSv1.2");
            } catch (NoSuchAlgorithmException e) {
                try {
                    // Fall back to TLS 1.1
                    System.out.println("Falling back to TLS 1.1");
                    context = SSLContext.getInstance("TLSv1.1");
                } catch (NoSuchAlgorithmException e2) {
                    // Fall back to TLS 1.0 if 1.2 and 1.1 are not 
                    // supported by any JSSE providers
                    System.out.println("Falling back to TLS 1.0");
                    context = SSLContext.getInstance("TLS");
                }
            }

            System.out.println("SSL/TLS provider: " + context.getProvider());

            // Create a KeyManagerFactory. 
            KeyManagerFactory kmf = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm(), "ENTRUST_JSSE");

            // Create and initialize a KeyStore.   The "Entrust" KeyStore provides 
            // a common interface to a variety of credential sources.  Here, 'epf' can
            // be the name of an Entrust EPF file or a PKCS#12 file (.p12 or .pfx).
            // See the javadoc for EntrustKeyStoreSpi and and the keystore sample code.
            // Alternatively, a "JKS" keystore is a standard JSSE Java KeyStore,
            // for working with outside keying material.            
            KeyStore eks = KeyStore.getInstance("Entrust");
            eks.load(new FileInputStream(epf), new SecureStringBuffer(new StringBuffer(pw)).toCharArray());

            // Initialize the KeyManagerFactory with this KeyStore
            kmf.init(eks, pw.toCharArray());

            // Create a TrustManagerFactory. 
            TrustManagerFactory tmf = TrustManagerFactory.getInstance(
                                            TrustManagerFactory.getDefaultAlgorithm(), "ENTRUST_JSSE");

            // Initialize the TrustManagerFactory with this KeyStore
            tmf.init(eks);

            // Initialize the SSLContext with the key manager and trust managers.
            // The key managers can be omitted if you do not want to do client
            // authentication, but trust managers must be present or else the
            // server certificate will be automatically rejected.
            context.init(kmf.getKeyManagers(),
                         tmf.getTrustManagers(),
                         Entrust.getDefaultSecureRandomInstance());

            // Get a socket factory and a socket from the SSLContext.
            SSLSocket socket = (SSLSocket) context.getSocketFactory().createSocket(host, port);

            if (includeSNI) {
                // RFC 6066 describes the Server Name Indication extension. The extension tells the
                // remote server which server the client wishes to connect to, and so
                // lets the remote server determine which server certificate to use in the
                // TLS handshake. (For example, if it is hosting multiple virtual servers; though
                // the SNI can be specified in non-virtual server scenarios as well.).
                //
                // Additionally, the client will perform a check of the hostname in the received
                // certificate against the hostname it is trying to connect to.
                //
                // The hostname specified in the SNI should be the fully qualified domain name (FQDN)
                // of the (virtual) server, as it is specified in the server's certificate
                // (i.e. the CN of the subject DN (or the Subject Alternative Name)).
                // For example, if the server is www.yourcompany.com, and the
                // subject DN is cn=www.yourcompany.com, o=yourcompany, c=US , then specify
                // 'www.yourcompany.com' for the SNI host name.  Or, if this is a
                // local test environment, and the certificate simply specifies
                // a local name, such as "cn=server1, o=yourcompany, c=US", then simply specify 'server1'.)
                // For this sample, the hostname will depend upon the Subject DN in the <EPF Name> used when
                // starting the JSSEServer sample.
                //
                // (Note, alternatively the SNIMatcher class can be used if a pattern match, as opposed to an
                //  exact match, is required.)
                System.out.println("Setting SNI extension to: " + host);
                SNIHostName serverName = new SNIHostName(host);
                List<SNIServerName> serverNames = new ArrayList<SNIServerName>(1);
                serverNames.add(serverName);

                SSLParameters params = socket.getSSLParameters();
                params.setServerNames(serverNames);

                // To enforce SNI checking on the client side, the endpoint identification algorithm must also be set.
                // Per the Java Security Standard Algorithm Names reference, valid values are "HTTPS" and "LDAPS".
                params.setEndpointIdentificationAlgorithm("HTTPS");
                socket.setSSLParameters(params);
            }

            // This socket can now be used like any other socket.  We'll send
            // a get message, and read the response.
            PrintWriter out = new PrintWriter(new BufferedWriter(new OutputStreamWriter(socket.getOutputStream())));
            out.println("GET / HTTP/1.0");
            out.println();
            out.flush();

            //  Print out the response from the server.
            InputStream is = socket.getInputStream();
            int r = is.read();

            while(r!=-1) {
                System.out.print((char)r);
                r=is.read();
            }

            System.out.println("Done.");

        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }
}
