//===========================================================================
//
// Copyright (c)  2000-2012 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.jsse;


import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.net.Socket;
import java.security.KeyStore;
import java.security.Security;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.TimeZone;
import java.util.Vector;

import javax.net.ServerSocketFactory;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLPeerUnverifiedException;
import javax.net.ssl.SSLServerSocket;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.TrustManagerFactory;
import javax.security.cert.X509Certificate;

import com.entrust.toolkit.security.provider.Entrust;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.jsse.provider.EntrustJSSEProvider;

/**
 * A simple example that shows how you can use the Toolkit to do SSL within the
 * JSSE API.  This is the server.
 * <p>
 * You can use the {@link JSSEClient} sample as the client, or you can use a web
 * browser.  In that case, after starting this server, point the browser to "https://localhost". 
 * <p>
 * If clientAuth is 'true' in the command-line arguments of this server sample, then the
 * server requires that clients should be authenticated, and you must import a .p12 file or
 * .pfx file into your web browser.  If you use the <CODE>JSSEClient</CODE> sample as 
 * the JSSE client, then the client's keystore file should contain the private key it needs.
 */
public class JSSEServer {
    
    // Initialize security providers and debug trace
    //
    static {
        // ALWAYS register the Toolkit's JCE providers.
        // Since this sample is demonstrating JSSE, initialize the Entrust JSSE Provider as well.
        Initializer.getInstance().setProviders(Initializer.MODE_ENTRUST_FIRST);
        Security.addProvider(new EntrustJSSEProvider());

        // Optional: Only when debugging, enable the trace from Sun's "SunJSSE"
        // provider.  Warning: Some secret data will be written to System.out !
        //System.setProperty("javax.net.debug", "ssl");
    }
    
    /**
     * main method
     */
    public static void main(String[] args) throws Exception {

        // Set the optional ClientAuth argument to true to require client authentication.
        // In that case, you should import a client .pfx or .p12 into your web browser,
        // or run the JSSEClient sample.
        if (args.length < 2) {
            System.out.println(
                "Usage: java JSSEServer <EPF Name> <Password> [<Port>] [<ClientAuth>]");
            return;
        }

        String epf = args[0];
        String pw = args[1];
        
        // Note: On Unix platforms, numbers less than 1024 are reserved ports, so you  
        //       might have to specify a higher port number on the command line.
        int port = (args.length < 3) ? 443 : Integer.parseInt(args[2]);
        boolean clientAuth = (args.length > 3) && (args[3].equals("true"));

        // Create an SSLServerSocket.
        ServerSocketFactory serverSocketFactory = init(epf, new SecureStringBuffer(new StringBuffer(pw))) ;
        SSLServerSocket serverSocket = (SSLServerSocket) serverSocketFactory.createServerSocket(port); 
        
        // Optional: Require clients to authenticate to this server.
        serverSocket.setNeedClientAuth(clientAuth);        
        
        // Start the server.
        System.out.println("Server will listen on port: " + serverSocket.getLocalPort());
        while (true) {
            
            Socket socket = null;
            InputStream is = null ;
            PrintStream ps = null ;
 
            try {
                socket = serverSocket.accept();
                
                System.out.println("Connection from: " + socket.getInetAddress().getHostName());
                
                //  Get streams that do SSL.
                is = new BufferedInputStream(socket.getInputStream());
                ps = new PrintStream(socket.getOutputStream());
                
                // If "SunJSSE" is the JSSE provider, the handshake happens here.
                Vector<String> request = readRequest(is);
                
                // An SSL session has been negotiated successfully.  Check whether
                // we shall allow this particular client to access resources on this server.
                String clientIdentity = checkAuthorization(serverSocket, socket) ;
                
                // As an example, just send a simple HTML page with info about this session.
                // Ignore what the client actually requested;  we're not a real server.
                sendSimplePage(ps, request, ((SSLSocket) socket).getSession().getCipherSuite(), clientIdentity);
                
                // Optional, just for testing, you might force the client to negotiate instead of resume.
                // ((SSLSocket) socket).getSession().invalidate();
            }
            catch (IOException e) {
                
                // SSL protocol allows sessions to be resumed or renegotiated, so this simple test
                // catches all IOExceptions and continues.  The test has succeeded when the peer
                // receives the simple HTML page sent by this server.
                e.printStackTrace();
            }
            
            if(is != null) {
                is.close();
            }
            if(ps != null) {
                ps.close();
            }
            if(socket != null) {
                socket.close();                  
            }           
        }
    }
    
    /**
     * Initialize a ServerSocketFactory with a key manager, trust manager, and JCE provider.
     */
    static private ServerSocketFactory init(String epfFilename, SecureStringBuffer pw)
    throws Exception
    {
        // Create an SSL context that supports the highest version of TLS supported
        // by the provider.   For the Oracle 1.7 JVM this will be TLS 1.2, older JVM's
        // only support TLS 1.0.
        SSLContext context = SSLContext.getInstance("TLS");
        System.out.println("SSL/TLS provider: " + context.getProvider());

        KeyManagerFactory kmf = 
            KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm(), "ENTRUST_JSSE");

        
        // Create and initialize a KeyStore.   The "Entrust" KeyStore provides 
        // a common interface to a variety of credential sources.  Here, 'epf' can
        // be the name of an Entrust EPF file or a PKCS#12 file (.p12 or .pfx)
        // or an Entrust KST key store ini file (.kst).
        // (See the javadoc for EntrustKeyStoreSpi and and the keystore sample code.)
        // Alternatively, a "JKS" keystore could be used to provide outside keying material.
        KeyStore eks = KeyStore.getInstance("Entrust");

        eks.load(new FileInputStream(epfFilename),  pw.toCharArray());
        kmf.init(eks, pw.toCharArray());

        
        // Initialize a TrustManagerFactory with this KeyStore.
        TrustManagerFactory tmf = TrustManagerFactory.getInstance(
                                            TrustManagerFactory.getDefaultAlgorithm(), "ENTRUST_JSSE");
        tmf.init(eks);                                                

        // Initialize the SSL context with key managers and trust managers.
        // If the trust managers are omitted, you will not be able to use
        // client authentication successfully.  Let's use a random number 
        // generator that is in accordance with FIPS 186-2.
        context.init(kmf.getKeyManagers(), tmf.getTrustManagers(),
                       Entrust.getDefaultSecureRandomInstance());
        return context.getServerSocketFactory();
    }
    
    /*
     * Read the request from the client
     */
    static private Vector<String> readRequest(InputStream is) throws IOException {
        
        Vector<String> request = new Vector<String>();
        BufferedReader reader = new BufferedReader(new InputStreamReader(is));
        while (true) {
            String line = reader.readLine();
            if ((line == null) || (line.length() == 0)) {
                break;
            }
            System.out.println(line);
            request.addElement(line);
        }

        return request;
    }
    
    /**
     * Write a page that has information about the SSL session.
     */
    static private void sendSimplePage(PrintStream ps, Vector<String> request, String ciphersuite, String clientName)
    {
        final SimpleDateFormat dateFormat = new SimpleDateFormat("EEE, d MMM yyyy HH:mm:ss z"); 
        dateFormat.setCalendar(new GregorianCalendar(TimeZone.getTimeZone("GMT")));
        
        // HTML headers
        ps.println("HTTP/1.0 200 OK");
        ps.println("Content-Type: text/html");
        ps.println("Server: MyJsseServer");
        ps.println("Date: " + dateFormat.format(new Date()));
        ps.println();
        
        // Content
        ps.println("<HTML><HEAD><TITLE>SSL Test</TITLE></HEAD>");
        ps.println("<BODY><H1>Success!</H1>");
        if (clientName != null) {
            ps.println("<P>Greetings to client<b> \"" + clientName + "\"</b></P>");
        }        
        ps.println("Your request was received on <b>" + new Date() + "</b> using ciphersuite <b>" + ciphersuite + "</b>");
        ps.println("<P>Your request was:<BLOCKQUOTE><PRE>");
        for (java.util.Enumeration<String> e = request.elements(); e.hasMoreElements();) {
            ps.println(e.nextElement());
        }
        ps.println("</PRE></BLOCKQUOTE></P></BODY></HTML>");        
    }
    
    /**
     * Get the name of the authenticated peer, and check whether is is authorized 
     * to access resources on this server.  This is only sample code, so all clients 
     * will be authorized;  the code just shows how you can get the identity of an
     * authenticated peer.
     * 
     * @throws IOException if the peer shall not be authorized
     */
    
    static private String checkAuthorization(SSLServerSocket serverSocket, Socket socket)
    throws IOException
    {
        String name = null ;
        if(!serverSocket.getNeedClientAuth()) {
            // If clients are not required to authenticate themselves, then they are authorized.
            return name ;
        }
       
        try {
            SSLSession session = ((SSLSocket) socket).getSession(); 
            X509Certificate[] peerCerts = session.getPeerCertificateChain();
            name = peerCerts[0].getSubjectDN().getName();
            if(!isAuthorized(name)) {
                throw new IOException("\"" + name + 
                                        " \"is not authorized to access resources on this server");
            }
        }
        catch (SSLPeerUnverifiedException e) {
            // We should never get here;  we should always be able to get the peer certificates.
            throw new RuntimeException("Somehow a client has been authenticated without sending its certificate!");
        }
        return name ;      
    }
    
    
    /**
     * Return true if 'clientName' is permitted to access resources on this server.
     * <P>
     * This is just a JSSE sample, so every authenticated client will be granted access.
     * In real life, authorization control can be fine grained and possibly complex.
     */
    static private boolean isAuthorized(String clientName)
    {
        return (clientName != null) && (clientName.length() > 0);
    }
 
  }