//===========================================================================
//
// Copyright (c)  2000-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.keystore;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.security.Key;
import java.security.KeyStore;
import java.security.cert.Certificate;

import com.entrust.toolkit.credentials.EntrustKeyStore;
import com.entrust.toolkit.credentials.EntrustKeyStoreSpi;
import com.entrust.toolkit.keystore.KSIniFileCreator;
import com.entrust.toolkit.util.SecureStringBuffer;

/**
 * This is a simple example of how the KeyStore can be used.
 * 
 * <p>
 * To run the example, an Entrust Profile (epf) and its password has to be 
 * provided.  The example creates a KeyStore-IniFile, and sets the given epf 
 * as the credential store in that KeyStore-IniFile.  Additionally, a 
 * read/write certificate store is set in that KeyStore-IniFile, with the 
 * password "Passw0rd".  Then, the KeyStore-IniFile is protected with the epf 
 * provided to this example.
 * </p>
 * 
 * <p>
 * In a second part, a login is done to the KeyStore using the KeyStore-IniFile 
 * that was created before using the epf's password.  The keys and certificates 
 * are read from the KeyStore and the signing and encryption certificate is 
 * written to the read/write certificate store.  At the end, the read/write 
 * certificate store is stored by calling the store() method of the KeyStore.  
 * Finally, the key store itself is closed using a special invocation of the 
 * store() method.
 * </p>
 */
public class KeyStoreExample {

    /** The path to the KeyStore INI file. */
    public static final String s_keyStoreIni = "KSIniFile.kst";

    /** The path were the read/write certificate store will be written. */
    public static final String s_readWriteCertificateStore = "ReadWriteCertStore.p12";

    /** Certificate store password. */
    public static final String s_certStorePassword = "Passw0rd";

    /**
     * Sample Application
     * <p></p>
     */
    public static void main(String args[]) {
        // Read the application arguments
        if (args.length != 2) {
            System.out.println("Usage: KeyStoreExample <epf> <password>");
            System.exit(1);
        }
        String epf = args[0];
        SecureStringBuffer password = new SecureStringBuffer(new StringBuffer(args[1]));

        // Login to the key store
        try {
            System.out.print("Creating KeyStore INI file '" + s_keyStoreIni + "'... ");
            KSIniFileCreator ksIniFileCreator = new KSIniFileCreator(epf);

            // Add read/write certificate stores
            SecureStringBuffer certStorePassword = new SecureStringBuffer(new StringBuffer(s_certStorePassword));
            ksIniFileCreator.addWriteCertificateStore(s_readWriteCertificateStore, certStorePassword);

            // Optionally, set a Directory connection
            //ksIniFileCreator.setLdap(localhost, 389);

            // Optionally, set an Authority connection
            //ksIniFileCreator.setAuthority(localhost, 829);
            
            // Create the key store INI file
            ksIniFileCreator.store(new FileOutputStream(s_keyStoreIni), password);
            System.out.println("DONE");
                
            // Login to the KeyStore
            System.out.print("Loading a KeyStore using the INI file '" + s_keyStoreIni + "'... ");
            KeyStore keyStore = KeyStore.getInstance("Entrust");
            keyStore.load(new FileInputStream(s_keyStoreIni), password.getStringBuffer().toString().toCharArray());
            System.out.println("DONE");

            // Extract the signing key from the KeyStore
            System.out.print("Extracting the signing key from the KeyStore... ");
            Key signingKey = keyStore.getKey(EntrustKeyStore.SIGNING_KEY, null);
            if (signingKey != null) {
                System.out.println("DONE");
            } else {
                System.out.println("NOT FOUND");
            }

            // Extract the decryption key from the KeyStore
            System.out.print("Extracting the decryption key from the KeyStore... ");
            Key decryptionKey = keyStore.getKey(EntrustKeyStore.DECRYPTION_KEY, null);
            if (decryptionKey != null) {
                System.out.println("DONE");
            } else {
                System.out.println("NOT FOUND");
            }
            
            // Extract the verification certificate from the KeyStore
            System.out.print("Extracting the verification certificate from the KeyStore... ");
            Certificate[] verificationCertChain = keyStore.getCertificateChain(EntrustKeyStore.VERIFICATION_CERTIFICATE);
            Certificate verificationCert = keyStore.getCertificate(EntrustKeyStore.VERIFICATION_CERTIFICATE);
            if (verificationCertChain != null && verificationCert != null) {
                System.out.println("DONE");
            } else {
                System.out.println("NOT FOUND");
            }

            // Extract the encryption certificate from the KeyStore
            System.out.print("Extracting the encryption certificate from the KeyStore... ");
            Certificate[] encryptionCertChain = keyStore.getCertificateChain(EntrustKeyStore.ENCRYPTION_CERTIFICATE);
            Certificate encryptionCert = keyStore.getCertificate(EntrustKeyStore.ENCRYPTION_CERTIFICATE);
            if (encryptionCertChain != null && encryptionCert != null) {
                System.out.println("DONE");
            } else {
                System.out.println("NOT FOUND");
            }

            // Since we didn't set a connection to the Directory, we can't get any 
            // certificates from the Directory.  However, if this was done, 
            // something similar to the following could be used to fetch 
            // certificates from the
            //System.out.println("Get some certificates from the directory! *******************************");
            //Certificate c1 = keyStore.getCertificate("cn=swprof test01, o=Entrust, c=CH");
            //Certificate c3 = keyStore.getCertificate("cn=swprof test03, o=Entrust, c=CH");
            //Certificate c6 = keyStore.getCertificate("cn=swprof test06, o=Entrust, c=CH");
            //Certificate c7 = keyStore.getCertificate("cn=swprof test07, o=Entrust, c=CH");
            //if(c1 != null && c3 != null && c6 != null && c7 != null) {
            //System.out.println("All certificates received from the directory!");
            //} else {
            //System.out.println("NOT all certificates received from the directory!");
            //}

            // Write encryption and signing certificates from our user to the 
            // read/write certificate store
            System.out.print("Writing signing and encryption certificates to the read/write KeyStore certificate store... ");
            keyStore.setCertificateEntry("encCert", encryptionCert);
            keyStore.setCertificateEntry("sigCert", verificationCert);
            System.out.println("DONE");

            // Read the encryption and signing certificates from the read/write
            // certificate store
            System.out.print("Reading signing and encryption certificates from the read/write KeyStore certificate store... ");
            if (keyStore.getCertificate("sigCert") != null && keyStore.getCertificate("encCert") != null) {
                System.out.println("DONE");
            } else {
                System.out.println("NOT FOUND");
            }

            // Store the read/write certificate store(s) by calling the store() method of the KeyStore
            System.out.print("Storing the read/write KeyStore certificate store to '" + s_readWriteCertificateStore + "'... ");
            keyStore.store(null, null);
            System.out.println("DONE");

            // Close the key store using a special invocation of the store() method
            System.out.print("Closing the KeyStore... ");
            keyStore.store(null, EntrustKeyStoreSpi.CLOSE_KEY_STORE);
            System.out.println("DONE");
        } catch (Exception e) {
            System.out.println("FAILED");
            e.printStackTrace();
        }
    }
}