//===========================================================================
//
// Copyright (c)  2000-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.ldaps;

import iaik.x509.X509Certificate;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.security.cert.CertificateException;
import java.util.ArrayList;

import javax.net.ssl.HandshakeCompletedEvent;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.credentials.UserStatus;
import com.entrust.toolkit.exceptions.CertificationException;
import com.entrust.toolkit.exceptions.UserBadPasswordException;
import com.entrust.toolkit.exceptions.UserFatalException;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.directory.HandshakeCompleteInfo;
import com.entrust.toolkit.x509.directory.JNDIDirectory;
import com.entrust.toolkit.x509.directory.LDAPSConfig;

/**
 * Sample to show how to configure and login online to an Entrust User using a directory connection 
 * that is protected by SSL (LDAP over SSL).
 * 
 * <p>
 * Usage:
 * <pre>
 * LDAPSLogin path_to_epf password [-dir ip_address port {certs 0..N}] | [-ini entrust.ini] 
 * </pre>
 * <dl>
 * <dt>path_to_epf</dt><dd>Path to Entrust Profile. For example, data/userdata/RSAUser1.epf</dd>
 * <dt>password</dt><dd>The password for the EPF. For example, ~Sample7~</dd>
 * <dt>-dir ip_address port {certs 0..N} </dt><dd>Login using address of a Directory to connect.  The
 * port in which to connect the directory must also be supplied.  A list of trusted SSL certs 
 * may optionally be supplied as well.  For example, -dir ldapserver.company.com 636 trustedCert1.cer</dd>
 * <dt>-ini entrust.ini </dt><dd>Login using an INI file to retrieve the settings.</dd>
 * </dl>
 * 
 * For example, to Login using an LDAPS directory with a single trusted certificate
 * <pre>
 * java -classpath ../lib/enttoolkit.jar;classes com.entrust.toolkit.examples.ldaps.LoginLDAPS Ldaps.epf password -dir myLdapsServer.com 636 /testfiles/test.cer 
 * </pre>
 * For example, to Login using an LDAPS directory using an entrust.in ifile
 * <pre>
 * java -classpath ../lib/enttoolkit.jar;classes com.entrust.toolkit.examples.ldaps.LoginLDAPS Ldaps.epf password -ini myentrust.ini 
 * </pre>
 * For example, to Login using anonymous SSL (all LDAPS servers are trusted)
 * <pre>
 * java -classpath ../lib/enttoolkit.jar;classes com.entrust.toolkit.examples.ldaps.LoginLDAPS Ldaps.epf password -dir myLdapsServer.com 636
 * </pre>
 * </p>
 */
public class LoginLDAPS
{
	/**
	 * The main execution of the program
	 * @param args path_to_epf password [-dir ip_address port {certs 0..N}] | [-ini entrust.ini]
	 */
    public static void main(String args[])
    { 
        //  Check the command-line arguments.  If they're not there, then
        // print out a usage message.
        if (args.length < 3) {
            System.out.println();
            System.out.println("Login with LDAPS enabled");
            System.out.println();
            System.out.println("usage:  LDAPSLogin path_to_epf password [-dir ip_address port {certs 0..N}] | [-ini entrust.ini] ");
            System.out.println();
            return;
        }
        try {
            //  Parse in the command-line arguments.
        	
        	LoginLDAPS ldaps = new LoginLDAPS();
        	
            int index = 0;
            String profile = args[index++];
            String password = args[index++];
            SecureStringBuffer securepassword = new SecureStringBuffer(password.toCharArray());
            String command = args[index++];
         
            
            //  Display the parameters
            System.out.println();
            System.out.println("profile: " + profile);
            System.out.println("password: " + password);
            
            Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);

            
            if (command.equalsIgnoreCase("-dir")) {
            	ldaps.login_DIR(args, index, profile, securepassword);
            }
          
            if (command.equalsIgnoreCase("-ini")) {
            	ldaps.login_INI(args, index, profile, securepassword);
            }
            
            HandshakeCompleteInfo info = HandshakeCompleteInfo.getHandshakeInfo();
            ArrayList list = info.getEventInfoList();
            System.out.println("Size: " + list.size());
            for (int i=0; i<list.size(); i++) {
            	HandshakeCompletedEvent event = (HandshakeCompletedEvent)list.get(i);
            	System.out.println(event.toString());
            }
            
            
        }
        
        catch (Exception e) {
            System.out.println();
            System.out.println("exception thrown:");
            e.printStackTrace();
            
            if (e instanceof CertificationException) {
                System.out.println();
                System.out.println("inner exception:");
                ((CertificationException)e).getInnerException().printStackTrace();
            }
        }
    }
    
    /**
     * Login using an entrust.ini file.  The ini file is read from the command line arguments and
     * read in using the {@link User#setConnections(java.io.InputStream)} method.
     *  
     * @param args the command line arguments
     * @param index the latest index of the command line arguments
     * @param profile the string profile name
     * @param password the password that has been secured in a SecureStringBuffer
     * 
     * @throws UserFatalException if a fatal error occurs during login
     * @throws UserBadPasswordException the password is not correct
     * @throws FileNotFoundException if the entrust.ini file could not be found
     * @throws CertificateException if encountered during login
     */
    public void login_INI(String args[], int index, String profile, SecureStringBuffer password) throws UserFatalException, UserBadPasswordException, FileNotFoundException, CertificateException {
    	
    	String ini = args[index++];
    	System.out.println("Ini: " + ini);
    	
    	 //  Log into an Entrust user, whose credentials we will use
        User user = new User();
        CredentialReader cr = new FilenameProfileReader(profile);
        
        user.setConnections(ini);
        
        System.out.println("login");
        user.login(cr, password);
        System.out.println("   done");
        UserStatus status = user.getStatus();
        System.out.println(status); 	
    	
    }
    
    /**
     * Login using LDAPS by configuring an directory, port and optional list
     * of trusted certificates. 
     * 
     * @param args the command line arguments
     * @param index the latest index of the command line arguments
     * @param profile the string profile name
     * @param password the password that has been secured in a SecureStringBuffer
     * 
     * @throws CertificateException if encountered during login
     * @throws IOException 
     * @throws UserFatalException if a fatal error occurs during login
     * @throws UserBadPasswordException the password is not correct
     */
    public void login_DIR(String args[], int index, String profile, SecureStringBuffer password) throws CertificateException, IOException, UserFatalException, UserBadPasswordException {
    	
    	String directory = args[index++];
    	String port = args[index++];
    	X509Certificate cert;
    	ArrayList certlist = new ArrayList();
    	
        System.out.println("directory: " + directory);
        System.out.println("port: " + port);
    	
    	while (index < args.length) {
             System.out.println("cert " + args[index] + " added");
             cert = new X509Certificate(new FileInputStream(args[index++]));
             certlist.add(cert);
        }
    	
    	X509Certificate trustedcerts[] = (X509Certificate[])certlist.toArray(new X509Certificate[]{});
    	
    	User user = new User();
        CredentialReader cr = new FilenameProfileReader(profile);
         
        JNDIDirectory dir= new JNDIDirectory(directory,Integer.parseInt(port));
         
        LDAPSConfig sslconfig = new LDAPSConfig(true);
        
        //If no trusted certs were added, use anonymous SSL by using an Anonymous SSL Socket Factory
        if (trustedcerts.length == 0) {
           try {
        	   
			  sslconfig.setSSLSocketFactory("AnonymousSSLSocketFactory");
		   } 
           catch (ClassNotFoundException e) {
			  e.printStackTrace();
		   }
        }
        else {
           sslconfig.addTrustCerts(trustedcerts);
        }
        dir.setLDAPSConfig(sslconfig);
         
        user.setConnections(dir, null);
         
        System.out.println("login");
        user.login(cr, password);
        System.out.println("   done");
        UserStatus status = user.getStatus();
        System.out.println(status); 	
    }
    

}


