//===========================================================================
//
// Copyright (c)  2019 Entrust.  All rights reserved.
//
//===========================================================================

package com.entrust.toolkit.examples.pkcs10;

import iaik.asn1.structures.ChoiceOfTime;
import iaik.x509.X509Certificate;
import iaik.x509.X509ExtensionException;
import iaik.x509.X509Extensions;
import iaik.x509.extensions.PrivateKeyUsagePeriod;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.cert.CertificateException;
import java.util.GregorianCalendar;

import com.entrust.toolkit.User;
import com.entrust.toolkit.asn1.crmf.OptionalValidity;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.EntrustP10CertReqInfo;
import com.entrust.toolkit.credentials.EntrustP10CertRetriever;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.exceptions.AuthorizationCodeException;
import com.entrust.toolkit.exceptions.EntrustPKIXCMPException;
import com.entrust.toolkit.exceptions.UserBadPasswordException;
import com.entrust.toolkit.exceptions.UserFatalException;
import com.entrust.toolkit.exceptions.UserNotLoggedInException;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.util.AuthorizationCode;
import com.entrust.toolkit.util.ManagerTransport;
import com.entrust.toolkit.util.SecureStringBuffer;

/**
 * <pre>
 * This sample demonstrates how to request for a short lived certificate.  By default, the minimum certificate
 * lifetime issued by Entrust Security Manager is 7 days. This sample shows how to obtain a certificate
 * that is less than 7 days, for example a certificate whose lifetime is in minutes.
 * To obtain certificates of different lifetimes (for example, hours or days), simply adjust the
 * <code>OptionalValidity</code> period in the sample code below, as needed.
 *
 * This sample shows how to:
 * 1. Generate an RSA keypair in Software
 * 2. Retrieve the public key in software
 * 3. Send the public key to the CA for certification, with the request specifying a 1 minute lifetime
 * 4. Retrieve the certified public key as an X509Certificate
 *
 * <p>
 * Prerequisites for issuing certificates with short validity using Entrust Authority(TM) Security Toolkit for Java(R):
 * <p>
 * 1. Use Entrust Authority(TM) Security Toolkit for Java(R) 8.0 patch 190183 or later.
 *
 * 2. The following setting must be placed in %AUTHDATDIR%\manager\entmgr.ini:
 *  [policy]
 *  EnableShortCertValidity=1
 *
 *  Restart Security Manager services after adding the above setting.
 *
 * 3. Users must not have any per-user lifetimes set.
 * </pre>
 *
 * Before this sample is run, a reference number and authorization code must have been obtained by
 * an administrator.  This could be done programmatically using the Administration Toolkit for Java, or it could
 * be done by using Security Manager Administration.  The Administration toolkit for Java contains sample code called
 * "User Create" which can be used to automate this function.
 * This sample also requires access to an Entrust Administrator's Profile.
 */
public class CreateShortLivedCertificate {

    /**
     * The main program.
     *
     * @param args
     *        Program arguments. See the help below for the expected command line arguments for this main method.
     * @throws AuthorizationCodeException
     * @throws UserFatalException
     * @throws IOException
     * @throws UserBadPasswordException
     * @throws CertificateException
     * @throws X509ExtensionException
     * @throws EntrustPKIXCMPException
     * @throws NoSuchAlgorithmException
     * @throws UserNotLoggedInException
     */
    public static void main(String args[]) throws AuthorizationCodeException, UserFatalException, IOException, UserBadPasswordException, CertificateException, X509ExtensionException, EntrustPKIXCMPException, NoSuchAlgorithmException, UserNotLoggedInException {

        if (args.length < 13) {
            System.out.println("Usage: CreateShortLivedCertificate <SM hostname> <SM CMP port> <Admin Profile> <Admin Profile Password> <refnum> <authcode> " +
                    "<certDefn> <year> <month> <day> <hour of day [0..23]> <validity lifetime in minutes [0..60]> <X.509 cert output file>");

            System.out.println("   where <SM hostname> is the hostname or IP Address of Entrust Security Manager");
            System.out.println("   where <SM CMP port> is the CMP port for Security Manager (usually 829)");
            System.out.println("   where <Admin Profile> is the location of the Administrator's Entrust Profile (.epf) file");
            System.out.println("   where <Admin Profile Password> is the password to the Administrator's Profile");
            System.out.println("   where <refnum> is the Reference Number for a newly created user, or recovered user in Security Manager");
            System.out.println("   where <authcode> is the Authorization Code for a newly created user, or recovered user in Security Manager");
            System.out.println("   where <certDefn> is the Certificate Definition to use for the certificate (for example, Verification or Encryption)");
            System.out.println("   where <year> is the year the certificate should be valid in");
            System.out.println("   where <month> is the month the certificate should be valid (1..12)");
            System.out.println("   where <day> is the day of the month the certificate should be valid in (1..31)");
            System.out.println("   where <hour of day> is the hour of the certificate validity (0..23)");
            System.out.println("   where <validity lifetime in minutes> is the number of minutes the certificate should be valid for (0..60)");
            System.out.println("   where <X.509 cert output file> is the name of a file where the certificate will be written to in DER encoded X.509 format");

            return;
        }

        String Manager = args[0];
        int SMPort = Integer.parseInt(args[1]);

        String AdminProfile = args[2];
        String AdminPassword = args[3];

        SecureStringBuffer refnum = new SecureStringBuffer(new StringBuffer(args[4]));
        AuthorizationCode authCode = new AuthorizationCode(new StringBuffer(args[5]));

        String certDefn = args[6];

        int year = Integer.parseInt(args[7]);
        int month = Integer.parseInt(args[8]);
        int day = Integer.parseInt(args[9]);
        int hour = Integer.parseInt(args[10]);
        int minutesLifetime = Integer.parseInt(args[11]);

        File outputCertFile = new File(args[12]);

        // The following performs a simple validation of the time parameters (month, day, hour, minute),
        // and does not perform an exact date validation.
        // The underlying Java classes used will perform the exact validation, and will automatically "roll forward"
        // any values (for example, if "February 29" was provided, then the GregorianCalendar Java class
        // will roll this forward to March 1 for non-leap years.)

        if (month < 1 || month > 12) {
            System.out.println("Given month must be between 1 and 12");
            return;
        }

        if (day < 1 || day > 31) {
            System.out.println("Given day must be between 1 and 31");
            return;
        }

        if (hour < 0 || hour > 23) {
            System.out.println("Given hour must be between 0 and 23 hours");
            return;
        }

        if (minutesLifetime < 0 || minutesLifetime > 60) {
            System.out.println("Given minute life span must be between 0 and 60 minutes");
            return;
        }


        System.out.println("Generating keys...");
        KeyPair kp;
        try {
            kp = generateRSAKeyinSoftware(2048);
        } catch (NoSuchProviderException e) {
            System.out.println("Could not find the provider: " + e.getMessage());
            return;
        }


        // Set the connection to the Security Manager (the Certificate Authority)
        ManagerTransport manager = new ManagerTransport(Manager, SMPort);

        // Login to the Admin User (offline)
        User adminUser = new User();
        CredentialReader cr = new FilenameProfileReader(AdminProfile);
        adminUser.login(cr, new SecureStringBuffer(new StringBuffer(AdminPassword)));
        System.out.println("Successful login to the Admin Profile");


        System.out.println("Creating the OptionalValidity structure");
        // Note, in the GregorianCalendar class, months are enumerated starting from 0,
        // so subtract 1 from the provided value (which has a range of 1-12)
        month = --month;
        GregorianCalendar calBefore = new GregorianCalendar(year, month, day, hour,0);
        GregorianCalendar calAfter = new GregorianCalendar(year, month, day, hour, minutesLifetime);

        // Add a PrivateKeyUsage
        PrivateKeyUsagePeriod pku = new PrivateKeyUsagePeriod(calBefore.getTime(),calAfter.getTime());
        X509Extensions extensions = new X509Extensions();
        extensions.addExtension(pku);

        // Create the OptionalValidity structure required for a short lived certificate
        // Set the ChoiceOfTime values to your requirements.  In this sample, certificates with a lifetime
        // in minutes are being requested, but any duration (supported by Security Manager policy) could
        // be requested (for example, days or months).
        ChoiceOfTime notBefore = new ChoiceOfTime(calBefore.getTime());
        ChoiceOfTime notAfter = new ChoiceOfTime(calAfter.getTime());
        OptionalValidity validity = new OptionalValidity(notBefore, notAfter);

        // Create the P10Certificate Request Info,  PKU can be added as an extension
        System.out.println("Creating the P10 Request with the public key from the P11 device");
        EntrustP10CertReqInfo requestInfo = new EntrustP10CertReqInfo(kp.getPublic(), certDefn, extensions);

        // Set the validity
        requestInfo.setOptionalValidity(validity);

        // Create the P10CertRetriever:
        EntrustP10CertRetriever certRetriever = new EntrustP10CertRetriever(manager,adminUser);

        // Retrieve the certificate
        System.out.println("Sending the P10 Request to Entrust Security Manager");
        certRetriever.retrieveUserCert(refnum, authCode, requestInfo);

        // Get the certificate and display it
        X509Certificate cert = requestInfo.getUserCertificate();
        System.out.println("\nRetrieved the User Certificate");
        System.out.println(cert);

        // Now get access to the Private key, if needed (from Oracle p11) and chain[]
        PrivateKey pk = kp.getPrivate();

        // Print out the certificate chain
        X509Certificate[] caChain = adminUser.getCaCertificateChain();
        X509Certificate[] fullChain = new X509Certificate[caChain.length+1];
        fullChain[0] = cert;
        for (int i=0; i<caChain.length;i++) {
            fullChain[i+1] = caChain[i];
            System.out.println("\n---Certificate Chain");
            System.out.println(fullChain[i+1]);
        }

        // Save the X.509 certificate to a file
        FileOutputStream fos = new FileOutputStream(outputCertFile);
        fos.write(cert.toByteArray());
        fos.close();
        System.out.println("\nCertificate in X.509 format written to: " + outputCertFile.getAbsolutePath());
    }

    /**
     * This method generates an RSA signing KeyPair in Software
     *
     * @param strength The strength of the RSA key to be generated.
     * @return A KeyPair object containing an RSAPublicKey and RSAPrivateKey
     * @throws NoSuchProviderException
     * @throws NoSuchAlgorithmException
     */
    private static KeyPair generateRSAKeyinSoftware(int strength) throws NoSuchAlgorithmException, NoSuchProviderException {

        Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);
        KeyPairGenerator kpgen = KeyPairGenerator.getInstance("RSA", "Entrust");
        kpgen.initialize(strength);

        KeyPair kp = kpgen.generateKeyPair();

        // Everything was okay, so return the keypair
        return kp;
    }


    /**
     * This method generates an RSA signing KeyPair on an HSM device using
     * Oracle P11 key generation
     * Note, this method is not used by the sample, but is provided for reference.
     *
     * @param strength The strength of the RSA key to be generated.
     * @return A KeyPair object containing an RSAPublicKey and a TokenRSAPrivateKey
     * @throws NoSuchProviderException
     * @throws NoSuchAlgorithmException
     */
    private static KeyPair generateRSAKeyOnOracleToken(int strength) throws NoSuchAlgorithmException, NoSuchProviderException {


        KeyPairGenerator kpgen = KeyPairGenerator.getInstance("RSA", "SunPKCS11-nCode");
        kpgen.initialize(strength);

        KeyPair kp = kpgen.generateKeyPair();

        // Everything was okay, so return the keypair
        return kp;
    }



//    /**
//     * This method generates an RSA signing KeyPair on an HSM device and returns the public and Private
//     * keys as a KeyPair object.  The public key will be an instance of an iaik.security.rsa.RSAPublicKey,
//     * the private key will be an instance of a TokenRSAPrivateKey (since it never leaves the HSM device).  The
//     * label used for the PrivateKey will be "Signing Key";
//     * This method is provided for reference.
//     *
//     * @param pkcs11LC The PKCS11LibraryConnection object, representing the connection to the PKCS11 Device
//     * @param sessionID The sessionID representing the currently opened session with the PKCS11 device
//     * @param hwSlot The slotID for the PKCS11 device
//     * @param strength The strength of the RSA key to be generated.
//     * @return A KeyPair object containing an RSAPublicKey and a TokenRSAPrivateKey
//     * @throws IOException if an error occurs when generated the RSA keys.
//     */
//    static KeyPair generateRSAKeyOnToken(PKCS11LibraryConnection pkcs11LC, long sessionID, long hwSlot, int strength) throws IOException {
//
//        JNIPKCS11 tokenHandle = null;
//        KeyPair keyPair = null;
//        try {
//            tokenHandle = pkcs11LC.getJNIPKCS11();
//
//            try {
//                long handle[] = tokenHandle.createRSASigningKeys(sessionID, strength);
//                long publicKey = handle[0];
//                long privateKey = handle[1];
//                byte[] modulus = tokenHandle.getRSAPublicKeyModulus(sessionID, publicKey);
//                byte[] exponent = tokenHandle.getRSAPublicKeyPublicExp(sessionID, publicKey);
//                RSAPublicKey rsapublic = new RSAPublicKey(new BigInteger(1, modulus), new BigInteger(1, exponent));
//                TokenRSAPrivateKey tokenprivate = new TokenRSAPrivateKey(pkcs11LC, sessionID, privateKey, hwSlot );
//                keyPair = new KeyPair(rsapublic, tokenprivate);
//            }
//            catch (Exception e) {
//                throw new InvalidKeyException("RSA key gen on token has failed: " + e.getMessage());
//            }
//        }
//
//        catch (InvalidKeyException e) {
//            throw new IOException("InvalidKeyException occured when trying to generate keys: " , e);
//        } catch (UserFatalException e) {
//            throw new IOException("UserFatalException occured when trying to generate keys: ", e);
//        }
//
//        //Everything was okay, so return the keypair
//        return keyPair;
//    }
}
