//===========================================================================
//
// Copyright (c)  2000-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.pkcs7;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.InputStream;
import java.security.PrivateKey;

import iaik.x509.X509Certificate;
import iaik.x509.extensions.KeyUsage;

import com.entrust.toolkit.KeyAndCertificateSource;
import com.entrust.toolkit.PKCS7DecodeStream;
import com.entrust.toolkit.PKCS7EncodeStream;
import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.CertVerifier;
import com.entrust.toolkit.x509.policies.ClientSettings;

/**
 * Sample to show use of PKCS7EncodeStream and PKCS7DecodeStream with non-repudiation
 * keys. Non-repudiation keys are generally considered more sensitive than regular
 * signing keys. Whenever such a key is used, the owner of the key should be
 * made aware of exactly what is being signed, meaning non-repudiation keys
 * should not be used in session-based protocols such as SSL. For simplicity, this sample
 * does not present what will be signed to the user.
 * 
 * <p>
 * Usage:
 * <pre>
 * Pkcs7NonRepudiation path_to_epf password root_cert_file in_file
 * </pre>
 * <dl>
 * <dt>path_to_epf</dt><dd>Path to EPF file. For example, data/userdata/NonrepudiationIdentity1.epf</dd>
 * <dt>password</dt><dd>The password for the EPF. For example, ~Sample7~</dd>
 * <dt>root_cert_file</dt><dd>The file containing the root CA certificate. For example, data/userdata/SampleRoot.cer</dd>
 * <dt>in_file</dt><dd>The file to protect. For example, data/testfiles/test.txt</dd>
 * </dl>
 * 
 * For example, from the examples directory, 
 * <pre>
 * java -classpath ../lib/enttoolkit.jar;classes com.entrust.toolkit.examples.pkcs7.Pkcs7NonRepudiation
 *   data/userdata/NonrepudiationIdentity1.epf ~Sample7~ data/userdata/SampleRoot.cer data/testfiles/test.txt
 * </pre>
 * </p>
 */
public class Pkcs7NonRepudiation
{
    /**
     * Runs the Pkcs7NonRepudiation example.
     * 
     * @param args
     *     The program arguments. See class documentation for details on running this example.
     */
    public static void main(String args[])
    {
        //  Check the command line arguments.
        if (args.length != 4)
        {
            System.out.println();
            System.out.println("Signs a file using a non-repudiation key, then verifies the signature.");
            System.out.println();
            System.out.println(
                "usage: Pkcs7NonRepudiation <profile> <password> <file containing root certificate> <input file>");
            System.out.println();
            return;
        }
        try
        {
            //  Parse the command line arguments.
            int index = 0;
            String profile = args[index++];
            String password = args[index++];
            String caCertFile = args[index++];
            String inputFile = args[index++];

            //  Display the parameters
            System.out.println();
            System.out.println("profile: " + profile);
            System.out.println("password: " + password);
            System.out.println("Root CA certificate file: " + caCertFile);
            System.out.println("input file: " + inputFile);
            System.out.println();

            // Initialize the crypto algorithms
            Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);

            // Create the root certificate now so we know it's good before continuing.
            InputStream certFileStream = new FileInputStream(caCertFile);
            X509Certificate rootCertificate = new X509Certificate(certFileStream);

            // Create the data input stream now so we know it's good before continuing.
            InputStream dataStream = new FileInputStream(inputFile);

            // Log into an Entrust user, whose keys will be used
            // to sign the PKCS7.
            User user = new User();
            CredentialReader credentialReader = new FilenameProfileReader(profile);
            System.out.println("Logging in");
            user.login(credentialReader, new SecureStringBuffer(new StringBuffer(password)));
            System.out.println("Login complete");

            // Create a CertVerifier for creating and verifying the message.
            // Assumes no connection to a Directory, but in reality this should
            // be done with a Directory connection so that up to date revocation
            // information is available. 
            CertVerifier certVerifier = new CertVerifier(rootCertificate, null, new ClientSettings());

            // Set up a key and certificate source that will use the logged
            // in user's non-repudiation key, if one is available.
            KeyAndCertificateSource keyAndCertificateSource = new KeyAndCertificateSource(certVerifier);

            // Set the certificate chain to be used.
            keyAndCertificateSource.setCaChain(user.getCaCertificateChain());

            // To find the non-repudiation keys, first look for certificates with
            // that key usage that belong to the user.
            PrivateKey signingKey = null;
            X509Certificate verificationCertificate = null;
            KeyUsage keyUsage = new KeyUsage(KeyUsage.nonRepudiation);
            X509Certificate[] nonRepudiationCerts = user.getUserCertificates(keyUsage);

            if (nonRepudiationCerts.length > 0)
            {
                // If more than one non-repudiation certificate is available,
                // there should be some way to choose which certificate to use,
                // perhaps through a GUI mechanism. For simplicity we'll just take
                // the first one.
                PrivateKey nonRepudiationKey = user.getUserPrivateKey(nonRepudiationCerts[0]);
                if (nonRepudiationKey != null)
                {
                    System.out.println("User has non-repudiation key available, using that to sign");
                    signingKey = nonRepudiationKey;
                    verificationCertificate = nonRepudiationCerts[0];
                }
            }

            // If there's no non-repudiation key, fall back to the default signing key.
            if (signingKey == null)
            {
                System.out.println("No non-repudiation key available, using default signing key");
                signingKey = user.getSigningKey();
                verificationCertificate = user.getVerificationCertificate();
            }

            // Set the key to use for signing
            keyAndCertificateSource.setSigningInfo(signingKey, verificationCertificate);

            // For this example, the PKCS7 signed data is written to memory.
            // In practice this may be written to a file or a socket.
            ByteArrayOutputStream p7OutStream = new ByteArrayOutputStream();

            // Create the PKCS7 encoder object, setting it to sign and supplying
            // the initialized key and certificate source.
            System.out.println("Signing beginning");
            PKCS7EncodeStream encoder =
                new PKCS7EncodeStream(keyAndCertificateSource, p7OutStream, PKCS7EncodeStream.SIGN_ONLY);

            // Set the block size to use when encoding so that large data will be streamed.
            // This won't work so well here because a ByteArrayOutputStream is being used,
            // but this is a good idea if writing to a real stream.
            encoder.setBlockSize(4096);

            // Read all of the input data file, and write it to the PKCS7 encode stream.
            byte[] inputDataBytes = new byte[4096];
            int bytesRead = dataStream.read(inputDataBytes);
            while (bytesRead != -1)
            {
                encoder.write(inputDataBytes, 0, bytesRead);
                bytesRead = dataStream.read(inputDataBytes);
            }

            // Close the encoder to flush any data remaining in the stream, and write the signature.
            encoder.close();
            System.out.println("Signing complete.");
            
            // Before going on, log the user out as we no longer need the keys.
            System.out.println("Logging user out.");
            user.logout();

            // Now that the PKCS7 signed data has been created, let's verify it.
            // Create a new KeyAndCertificateSource for this. The CertVerifier can
            // be reused, or a new one could be created if there are special verification
            // needs, such as setting acceptable policy OIDs.
            System.out.println("Beginning signature validation.");
            keyAndCertificateSource = new KeyAndCertificateSource(certVerifier);

            // Turn the data that was written in to an InputStream              
            ByteArrayInputStream p7InStream = new ByteArrayInputStream(p7OutStream.toByteArray());

            // Create the PKCS7 decoder object.  This time we only supply the certificate verifier.
            PKCS7DecodeStream decoder = new PKCS7DecodeStream(keyAndCertificateSource, p7InStream);

            // Read all of the signed data from the decoder
            System.out.println("Following is the data that was signed:");
            bytesRead = decoder.read(inputDataBytes);
            while (bytesRead != -1)
            {
                // You'll probably do something more useful than just printing the signed data. 
                System.out.print(new String(inputDataBytes, 0, bytesRead));
                bytesRead = decoder.read(inputDataBytes);
            }

            // Close the decoder, which causes the signature to be validated.
            decoder.close();
            System.out.println();
            System.out.println("Signature validation complete.");
        }
        catch (Exception e)
        {
            // The exception handling here is deliberately vague
            // to make the sample more readable.
            System.out.println();
            System.out.println("Exception thrown:");
            e.printStackTrace();
        }
    }
}
