//===========================================================================
//
// Copyright (c)  2006-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.resourcemonitor;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;

import com.entrust.toolkit.resourcemonitor.Resource;
import com.entrust.toolkit.resourcemonitor.ResourceEvent;
import com.entrust.toolkit.resourcemonitor.ResourceEventProcessor;
import com.entrust.toolkit.resourcemonitor.ResourceMonitor;
import com.entrust.toolkit.resourcemonitor.ResourceOutputHandler;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.util.ManagerTransport;

/**
 * ManagerTransportEvent is a simple example that shows how the
 * Resource monitoring capabilities of the Java toolkit can be used
 * to monitor the {@link ManagerTransport} object.  It consists of
 * 5 example tests:
 * <ul>
 *    <li><b>SimpleMonitorExample</b> - The most basic way to setup polling
 *     for a ManagerTransport object is to call its periodicPoll()
 *     method</li>
 *    <li><b>SimpleEventHandlerExample</b> - Shows example of a user-defined
 *    event handler object.</li>
 *    <li><b>SimpleProcessorExample</b> - Shows how to setup polling using
 *    ResourceEventProcessing and only allow an event which is
 *    set by the ResourceEventProcessor.  Uses the SampleEventHandler</li>
 *    <li><b>ResourceMonitorExample</b> - Shows how to setup polling using
 *    a ResourceMonitor to monitor the Resource.  A ResourceMonitor can
 *    be used to monitor many resources at the same time.  Event notification
 *    is written to a file called ResourceMonitorExample.log</li>
 *    <li><b>MultipleHandlerExample</b> - Shows how more than one event handler
 *    can be attached to a Resource.  In this case, the event is handled
 *    by writing the contents of the event to System.out and a file called
 *    MultipleHandlerExample.log</li>  
 * </ul> 
 */

public class ManagerTransportEvent {

	public ManagerTransportEvent() {
	}

	/**
	 * @param args
	 */
	public static void main(String[] args) {

		if (args.length < 2) {
			System.out.println();
			System.out.println("Example of network monitoring");
			System.out.println();
			System.out.println("usage: ManagerTransportEvent <manager ip> <manager port>");
			return;
		}
		try {
			// Parse in the command-line arguments.
			int index = 0;
			String manip = args[index++];
			String manport = args[index++];

			// Display the parameters
			System.out.println();
			System.out.println("manager ip: " + manip);
			System.out.println("manager port: " + manport);

			//Setup Security Providers
			Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);
			
			//1. First demonstrate generic Usage of Resource Monitor...

			// Setup ManagerTransport object for Monitoring.  
			// ManagerTransport implements the PollableResource interface 
			ManagerTransport man = new ManagerTransport(manip, manport);
			
			ManagerTransportEvent events = new ManagerTransportEvent();
			
			//Run examples
			System.out.println("Running SimpleMonitorExample...");
			events.SimpleMonitorExample(man);
			System.out.println("\nRunning SimpleEventHandlerExample...");
			events.SimpleEventHandlerExample(man);
			System.out.println("\nRunning SimpleProcessorExample...");
			events.SimpleProcessorExample(man);
			System.out.println("\nRunning ResourceMonitorExample (Results are contained in ResourceMonitorExample.log)...");
			events.ResourceMonitorExample(man);	
			System.out.println("\nRunning MultipleHandlerExample...");
			events.MultipleHandlerExample(man);
			
            
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * This method shows the most basic way that polling can be done on a ManagerTransport
	 * object.  Simply calling the ManagerTransport.periodPoll() with the specified period
	 * in seconds with an associated EventHandler will enable polling for this Resource.  If
	 * no event handler is specified, a default ResourceOutputhandler is used which simply
	 * sends the event to System.out.
	 * The third option specifies the trigger point at which a {@link ResourceEvent}
	 * triggers EventNotification.  In this case all events trigger a notification.
	 * 
	 * @param man The ManagerTransport Object
	 * @throws InterruptedException
	 */
	public void SimpleMonitorExample(ManagerTransport man) throws InterruptedException {
		ResourceMonitor mon = man.periodicPoll(1,null,ResourceEvent.OKAY);
		Thread.sleep(15000);
		mon.stop();
		
//		Give monitor time to stop thread
		Thread.sleep(2000);
	}
	
	/**
	 * This method shows the most basic way that polling can be done on a ManagerTransport
	 * object using a user-defined event handler.  Simply calling the ManagerTransport.periodPoll()
	 * with the specified period in seconds, with a user-defined event handler will enable polling
	 * for this Resource. The third option specifies the trigger point at which a {@link ResourceEvent}
	 * triggers EventNotification.  In this case only a failure will trigger a notification.
	 * @param man The ManagerTransport Object
	 * @throws InterruptedException
	 */
	public void SimpleEventHandlerExample(ManagerTransport man) throws InterruptedException {
		SampleEventHandler sample = new SampleEventHandler();
		ResourceMonitor mon = man.periodicPoll(2,sample,ResourceEvent.FAILURE);
		Thread.sleep(15000);
		mon.stop();
		
        //Give monitor time to stop thread
		Thread.sleep(2000);
	}
	
	/**
	 * This method shows how the {@link ResourceEventProcessor} can be used to setup polling
	 * for any object which implements the PollableResource interface.  The ResourceEventProcessor
	 * is a listener that can be used to listen to Resource polling events.
	 * 
	 * @param man the ManagerTransport Object
	 * @throws InterruptedException
	 * @throws FileNotFoundException
	 */
	public void SimpleProcessorExample(ManagerTransport man) throws InterruptedException, FileNotFoundException {

        //Create the resource giving the polling period of seven seconds
		//using ManagerTransport 
		Resource resource = new Resource(3, man);
		resource.setEventTrigger(ResourceEvent.OKAY);
		
		//Each resource should have a ResourceListener attached so that
		// it is notified of polling events.  In this case, the 
		// ResourceEventProcessor class can be used to listen to all Resources
		ResourceEventProcessor eventprocess = new ResourceEventProcessor();
		
		//Set an event handler in this EventProcessor
		eventprocess.setEventHandler(new SampleEventHandler());
		
		//Force usage of this event handler
		eventprocess.setProcessingMode(ResourceEventProcessor.LOCAL_HANDLER_ONLY);

		// Register the ResourceEventProcessor with the resource.  
		resource.addResourceListener(eventprocess);
	    
        //start the resource
		resource.start();
	    
		// Sleep for 15 seconds, polling should continue....
		Thread.sleep(15000);
      
		//Stop monitoring 
		resource.stop();
		
//		Give monitor time to stop thread
		Thread.sleep(2000);
	}
	
	/**
	 * This method shows how the resource monitor framework can be used to setup polling
	 * for any object which implements the PollableResource interface.  The ResourceMonitor
	 * is a listener that can be used to listen to Resource polling events, as well as 
	 * Managing any number of resources.
	 * @param man The ManagerTransport Object
	 * @throws InterruptedException
	 * @throws FileNotFoundException
	 */
	public void ResourceMonitorExample(ManagerTransport man) throws InterruptedException, FileNotFoundException {

        //Create the resource giving the polling period of seven seconds
		//using ManagerTransport 
		Resource resource = new Resource(5, man);
		resource.setEventTrigger(ResourceEvent.OKAY);
		
		//Each resource should have a ResourceListener attached so that
		// it is notified of polling events.  In this case, the 
		// ResourceMonitor class can be used to listen to all Resources
		ResourceMonitor monitor = new ResourceMonitor();

		// Register the Listener with the resource.  This can be done by
		// either calling the ResourceMonitor.addPolledResource() method
		// or the Resource.addResourceListener() method.
	    monitor.addResource(resource);
	    
	    //Add the appropriate EventHandler.  The ResourceOutputHandler can
	    //write the contents of the event message to a FileOutputStream or
	    //a PrintWriter.   
        monitor.setEventHandler(new ResourceOutputHandler(new FileOutputStream("ResourceMonitorExample.log")));
	
        // Each resource can be started independently, or the
		// resource monitor can be used to start/stop all resources.  If
	    // the resource monitor is used to start the resources, they must
	    // be added into the monitor using the addPolledResource.
		//resource.start();
	    
		monitor.start();  
		// Sleep for 20 seconds, polling should continue....
		Thread.sleep(20000);
      
		//Stop monitoring 
		monitor.stop();
		
		//Give monitor time to stop thread
		Thread.sleep(2000);
	}
	
	/**
	 * This method shows how multiple ResourceEventHandlers can be setup for a 
	 * single Resource.  In this sample, Handers are used to write the output
	 * to the System.out and a file called "managerStatus.log".
	 * @param man The ManagerTransport Object 
	 * @throws InterruptedException
	 * @throws FileNotFoundException
	 */
	public void MultipleHandlerExample(ManagerTransport man) throws InterruptedException, FileNotFoundException {

        //Create the resource giving the polling period of seven seconds
		//using ManagerTransport 
		Resource resource = new Resource(7, man);
		resource.setEventTrigger(ResourceEvent.OKAY);
		
		//Each resource should have a ResourceListener attached so that
		// it is notified of polling events.  In this case, the 
		// ResourceEventProcessor class can be used to listen to all Resources
		ResourceEventProcessor eventprocess = new ResourceEventProcessor();
		
		//Set up multiple Resource event handlers for this resource
		resource.addEventHandler(new ResourceOutputHandler(System.out));
		resource.addEventHandler(new ResourceOutputHandler(new FileOutputStream("MultipleHandlerExample.log")));
		resource.addEventHandler(new SampleEventHandler());
		
		//Force usage of this event handler
		eventprocess.setProcessingMode(ResourceEventProcessor.RESOURCE_HANDLERS_ONLY);

		// Register the ResourceEventProcessor with the resource.  
		resource.addResourceListener(eventprocess);
	    
        //start the resource
		resource.start();
	    
		// Sleep for 29 seconds, polling should continue....
		Thread.sleep(29000);
      
		//Stop monitoring 
		resource.stop();
		
//		Give monitor time to stop thread
		Thread.sleep(2000);
	}
	
	
	
	
	
	
}
