//===========================================================================
//
// Copyright (c)  2003-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.roaming;

import iaik.x509.X509Certificate;

import java.security.PrivateKey;
import java.security.Signature;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialCreator;
import com.entrust.toolkit.credentials.CredentialReader;
import com.entrust.toolkit.credentials.CredentialWriter;
import com.entrust.toolkit.credentials.RoamingCredentialWriter;
import com.entrust.toolkit.exceptions.UserNotLoggedInException;
import com.entrust.toolkit.util.AuthorizationCode;
import com.entrust.toolkit.util.SecureStringBuffer;

/**
 * A sample application to demonstrate how to create a Roaming Digital
 * Identity. It accepts parameters user to create the identity with a
 * PKI, then creates that identity and sends it to the Roaming Server.
 * The Roaming Server will encrypt what is sent to it and place it in the
 * Directory.
 * 
 * <p>The following shows how to use this program:</p> 
 *
 * <pre>Create &lt;roaming user id&gt; &lt;password&gt; &lt;ref num&gt; &lt;auth code&gt; &lt;entrust.ini&gt;</pre>
 * 
 * <p>Below is an explanation of each parameter with example values:</p>
 * 
 * <dl>
 *      <dt>&lt;roaming user id&gt;</dt>
 *      <dd>The roaming user id that the user will use to log in</dd>
 *      <dt>&lt;password&gt;</dt>
 *      <dd>The password for the user, which the user will use for future logins</dd>
 *      <dt>&lt;ref num&gt;</dt>
 *      <dd>The Reference Number for the user that was obtained from the Security 
 *          Manager</dd>
 *      <dt>&lt;auth code&gt;</dt>
 *      <dd>The Authorization Code for the user that was obtained from the Security 
 *          Manager</dd>
 *      <dt>&lt;entrust.ini&gt;</dt>
 *      <dd>The path to the entrust.ini file that contains all of the settings
 *          required to communicate with the Security Manager and Roaming Server.</dd>
 * </dl>
 * 
 * <p>
 * After the user is created, a sign/verify test is run to make sure the user was created
 * correctly.
 * </p>
 * 
 * <dl>
 *     <dt>Note:</dt>
 *     <dd>This program requires an Entrust Authority Roaming Server 6.0 or higher
 *         to run.</dd>
 * </dl>
 * 
 */
public class Create
{

    /**
     * The main program.
     * 
     * @param argv
     *     The program arguments. See the class overview for a description.
     */
    public static void main(String argv[])
    {
        if (argv.length != 5)
        {
            System.out.println("Usage: java com.entrust.toolkit.examples.roaming.Create <roaming user id>");
            System.out.println("           <password> <ref num> <auth code> <entrust.ini>");
            return;
        }

        // Extract the command line arguments and display them.
        String roamingUserId = argv[0];
        String password = argv[1];
        String iniFile = argv[4];

        System.out.println();
        System.out.println("Roaming Sample application to demonstrate creating a Roaming Digital Identity");
        System.out.println("user id  = " + roamingUserId);
        System.out.println("password = " + password);
        System.out.println("reference number = " + argv[2]);
        System.out.println("authorization code = " + argv[3]);
        System.out.println("entrust.ini = " + iniFile);

        // Start by creating a basic User object.
        User roamingUser = new User();
        try
        {
            SecureStringBuffer referenceNumber = new SecureStringBuffer(new StringBuffer(argv[2]));
            AuthorizationCode authorizationCode = new AuthorizationCode(new StringBuffer(argv[3]));

            // Create a credential reader that creates the keys and certificates
            // To recover instead of creating, use the following code:
            //     CredentialReader reader = new CredentialRecoverer(referenceNumber, authorizationCode);
            CredentialReader reader = new CredentialCreator(referenceNumber, authorizationCode);

            // Set a credential writer so that the digital identity can be
            // written to the Roaming Server.
            CredentialWriter writer = new RoamingCredentialWriter(roamingUserId, iniFile);
            roamingUser.setCredentialWriter(writer);

            // Set connections so that the User object can contact the Directory and PKI.
            roamingUser.setConnections(iniFile);

            // now call the login method to actually create the user.
            System.out.println("Creating Roaming User...");
            roamingUser.login(reader, new SecureStringBuffer(new StringBuffer(password)));
            System.out.println("Creation complete");
            System.out.println();

            // Test Signing and verification with the newly created user.
            System.out.println("\n\nSign/Verify Test");
            PrivateKey signingKey = roamingUser.getSigningKey();
            X509Certificate verificationCertificate = roamingUser.getVerificationCertificate();
            Signature signer = Signature.getInstance("SHA-1/" + signingKey.getAlgorithm(), "Entrust");

            // Sign a sample message
            byte[] messageToSign = "Some message to sign".getBytes();
            System.out.println("\nAttempting sign operation using the user's signing key...\n");
            signer.initSign(signingKey);
            signer.update(messageToSign);
            byte[] signature = signer.sign();

            // Verify the message
            System.out.println(
                "\nAttempting verify operation using the user's verification certificate...\n");
            signer.initVerify(verificationCertificate);
            signer.update(messageToSign);
            boolean verified = signer.verify(signature);
            if (verified)
            {
                System.out.println("Signature verification was successful");
            }
            else
            {
                System.out.println("Signature verification FAILED!");
            }
        }
        catch (Exception e)
        {
            // No error handling is performed to keep the code simpler.
            e.printStackTrace();
        }
        finally
        {
            // ensure that the user is always logged out at the end of the program
            try
            {
                roamingUser.logout();
            }
            catch (UserNotLoggedInException e)
            {
                // it's safe to ignore this Exception at this point in the program
            }
        }
    }

}
