//===========================================================================
//
// Copyright (c)  2003-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.roaming;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.CredentialWriter;
import com.entrust.toolkit.credentials.FilenameProfileWriter;
import com.entrust.toolkit.credentials.RoamingCredentialReader;
import com.entrust.toolkit.exceptions.UserNotLoggedInException;
import com.entrust.toolkit.util.SecureStringBuffer;

/**
 * A sample application to demonstrate how to remove a Roaming Digital
 * Identity from the Directory, through the process known as "deregistration".
 * It accepts a roaming user id, a password, and a path to an
 * entrust.ini file, then logs in to that identity, writes it to a file on
 * disk, and then requests the Roaming Server deregister the user.
 * 
 * <dl>
 *     <dt>Note:</dt>
 *     <dd>This program requires an Entrust Authority Roaming Server 6.0 or higher
 *         to run.</dd>
 * </dl>
 */
public class Deregister
{

    /**
     * The main program.
     * 
     * @param argv
     *     The program arguments. Three program arguments are required:
     * <ol>
     * <li>roaming user id - the roaming user id to log in to</li>
     * <li>password - the user's password</li>
     * <li>entrust.ini - the path to the entrust.ini file containing the roaming configuration settings.</li>
     * </ol>
     */
    public static void main(String argv[])
    {
        if (argv.length != 3)
        {
            System.out.println(
                "Usage: java com.entrust.toolkit.examples.roaming.Deregister <roaming user id> <password> <entrust.ini>");
            return;
        }

        // Extract the command line arguments and display them.
        String roamingUserId = argv[0];
        String password = argv[1];
        String iniFile = argv[2];

        System.out.println();
        System.out.println(
            "Roaming Sample application to demonstrate removal of a Roaming Digital Identity from the Directory");
        System.out.println("user id  = " + roamingUserId);
        System.out.println("password = " + password);
        System.out.println("entrust.ini = " + iniFile);

        // Start by creating a basic User object.
        User roamingUser = new User();
        try
        {
            // Create a credential reader that will read the roaming credentials
            // for user roamingUserId. iniFile must contain all of the required
            // roaming configuration information.
            RoamingCredentialReader reader = new RoamingCredentialReader(roamingUserId, iniFile);

            // Set a credential writer so that the digital identity can be
            // written and is not lost. It is very important to do this, otherwise
            // the digital identity will be lost and must be recovered.
            CredentialWriter writer = new FilenameProfileWriter(roamingUserId + ".epf");
            roamingUser.setCredentialWriter(writer);

            // Set connections so that the User object can contact the Directory and PKI.
            roamingUser.setConnections(iniFile);

            // Perform the roaming login
            System.out.println("Roaming Login...");
            roamingUser.login(reader, new SecureStringBuffer(new StringBuffer(password)));
            System.out.println("Login complete");
            System.out.println();

            // Force a write operation to make sure the identity is written out.
            System.out.println("Writing the identity to a file");
            System.out.println();
            roamingUser.write();

            // Now request the server remove all roaming files from the Directory.
            // Note that if this roaming identity was used with the Entrust Desktop software,
            // it is better to let the Desktop software perform this operation as it
            // will not be able to find the disk-based identity.
            System.out.println("Sending deregistration request to Roaming Server");
            System.out.println();
            reader.deregister();

            System.out.println("Successfully deregistered roaming user '" + roamingUserId + "'");
        }
        catch (Exception e)
        {
            // No error handling is performed to keep the code simpler.
            e.printStackTrace();
        }
        finally
        {
            // ensure that the user is always logged out at the end of the program
            try
            {
                roamingUser.logout();
            }
            catch (UserNotLoggedInException e)
            {
                // it's safe to ignore this Exception at this point in the program
            }
        }
    }
}
