//===========================================================================
//
// Copyright (c)  2001-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.smime;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.exceptions.UserBadPasswordException;
import com.entrust.toolkit.exceptions.UserNotLoggedInException;
import com.entrust.toolkit.util.IniFile;
import com.entrust.toolkit.util.ManagerTransport;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.directory.JNDIDirectory;

import iaik.asn1.ObjectID;
import iaik.asn1.structures.AlgorithmID;
import iaik.asn1.structures.Name;
import iaik.pkcs.PKCSException;
import iaik.pkcs.pkcs10.CertificateRequest;
import iaik.security.smime.EncryptedContent;
import iaik.security.smime.PKCS10Content;
import iaik.security.smime.SMimeBodyPart;
import iaik.security.smime.SMimeMultipart;
import iaik.security.smime.SMimeParameters;
import iaik.security.smime.SignedContent;
import iaik.x509.X509Certificate;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URL;
import java.security.InvalidKeyException;
import java.security.KeyException;
import java.security.PrivateKey;
import java.security.SignatureException;
import java.util.Date;
import java.util.Properties;

import javax.activation.CommandMap;
import javax.activation.DataHandler;
import javax.activation.MailcapCommandMap;
import javax.activation.URLDataSource;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;

/**
 * This class demonstrates how the IAIK S/MIME implementation and the Entrust 
 * Toolkit work together efficiently.
 * To run this demo you also need the packages:
 * <ul>
 * <li>mail.jar: Get it from <a href="http://www.javasoft.com/products/javamail/">Sun</a>.
 * <li>activation.jar: Get it from <a href="http://java.sun.com/beans/glasgow/jaf.html">Sun</a>.
 * <li>OPTIONAL: pop3.jar: If you need POP3 access. 
 *     Get it from <a href="http://www.javasoft.com/products/javamail/">here</a>.
 * </ul>
 */
public class SMimeSend {
    
    /**
	 * Array of signer's verification certificate and CA certificate
	 */
   X509Certificate[] signerCertificates = null;         
  /**
   * encryption certificate of the recipient
   */
   X509Certificate recipientCertificate = null;    
  /**
   * verification certificate of the signer
   */
  X509Certificate signerCertificate = null;       
  /**
   * encryption certificate of the signer
   */
  X509Certificate encryptionCertOfSigner = null;  
  /**
   * private key of the signer
   */
  PrivateKey signerPrivateKey = null;
  
  IniFile properties = null;
   
  String section = null;
  String attrToFind = null; 
  String searchbase = null;          
  String searchexpr = null;    
  String to = null;                     
  String from = null;                 
  String host = null;                
  String Profile = null;               
  String password = null;         
  String pki = null;
  String port = null;
  String firstName = null;
  String lastName = null;
  String mailcapFile = null;
  
  
  /**
   * Creates a new SMimeSend object
   */
  public SMimeSend() throws Exception {
 
    //get the properties from example.ini
    try{
    properties = new IniFile("data/smime/smime_example.ini");
    }
    catch(FileNotFoundException ex){
        System.out.println("Could not found the file data/smime/smime_example.ini: " + ex.getMessage());
    }
    section = new String("SMimeSend");
    //Search criteria for Recipient
    attrToFind = properties.getString(section,"attrToFind"); 
    //to edit begin
    searchbase = properties.getString(section,"searchbase");         
    searchexpr = properties.getString(section,"searchexpr");     
    to = properties.getString(section,"to");                          
	from = properties.getString(section,"from");                 
	host = properties.getString(section,"host");                
	//Profile of Sender
	Profile = properties.getString(section,"Profile");               
	password = properties.getString(section,"password");         
	//to edit end
    // path to mailcap file, defaults to "mailcap"    
    mailcapFile = properties.getString(section,"mailcap", "mailcap");
    
    //now we read the profile of the sender into an Inputstream
    FilenameProfileReader fis = null;
    try {
        fis = new FilenameProfileReader(Profile); 
    }
    catch(FileNotFoundException e) {
        System.out.println("Profile not found!");
        throw e;
    }
   
    User sender = new User();
    try {
        //connect to the JNDIDirectory ->get information from the entrust.ini File
        String[] info = getPKIadress();
        pki = info[0];
        port = info[1];
        if( info[0] != null && info[1] != null )
        {
            JNDIDirectory dir = new JNDIDirectory(pki, (new Integer(port)).intValue());
            ManagerTransport emt = new ManagerTransport(pki, 829);
    
            sender.setConnections(dir, emt);
            sender.login(fis, new SecureStringBuffer(password));
            
            
            //get the recipient certificate from the directory
            byte[][] result = dir.Search(searchbase,searchexpr,attrToFind);
            recipientCertificate = new X509Certificate(result[0]);
        }
        else
        {
            String certificateFile = properties.getString(section,"recipientCertificate"); 
            System.out.println("Working offline, using certificate file " + certificateFile + " as recipient");
            recipientCertificate = new X509Certificate(new FileInputStream(certificateFile));
            sender.login(fis, new SecureStringBuffer(password));
        }
    }
    catch(Exception e) {
        System.out.println("Login failed: " + e.getMessage());
        throw e;
    }
    //lets check if recipientCertificate is  a valid certificate, i.e., there is a valid certificate path from recipientCertificate  
    //to the root of trust.
    try{ 
            sender.validate(recipientCertificate);
            System.out.println("Recipient's Certificate trusted!");
           }
        catch(Exception ex){
            System.out.println("Recipient's Certificate not trusted!");
        }
   
    
    X509Certificate[] certs = new X509Certificate[2];
    try {
        certs[0] = sender.getVerificationCertificate();
        certs[1] = sender.getCaCertificate();    
    
        signerCertificates = certs;
        signerPrivateKey = sender.getSigningKey();
        signerCertificate = sender.getVerificationCertificate();
    
        // recipient = signer for this test
        encryptionCertOfSigner = sender.getEncryptionCertificate();
        System.out.println("Algorithm :" + sender.getEncryptionCertificate().getPublicKey().getAlgorithm());
    }
    catch(UserNotLoggedInException e) {
        System.out.print(e.getMessage());
        System.out.print("User not logged in!");
    }

    for (int i = 0; i < signerCertificates.length; i++) 
      System.out.println(signerCertificates[i].toString(true));
    System.out.println(recipientCertificate.toString(true));
    System.out.println(encryptionCertOfSigner.toString(true));
  }    

  /**
   * starts to send ten different kinds of messages from plain messages to signed and encrypted messages with different 
   * encryption and hash algorithm
   * 
   * @exception java.io.IOException
   */
  public void start() throws IOException {
    
	// register content data handlers for S/MIME types
	MailcapCommandMap mc = new MailcapCommandMap(mailcapFile);
	CommandMap.setDefaultCommandMap(mc);

    // create some properties and get the default Session
  	Properties props = new Properties();
  	props.put("mail.smtp.host", host);
  	//if you like to see more use debug mode
	//props.put("mail.debug", "true");
  	Session session = Session.getDefaultInstance(props, null);

  	try {
      // Create a demo Multipart
      MimeBodyPart mbp1 = new SMimeBodyPart();
	  mbp1.setText("This is a Test of the IAIK S/MIME implementation!\n\n");
	  // try to test an attachment
	  // this demo attaches our homepage
      MimeBodyPart attachment = new SMimeBodyPart();
      URL url = new URL("http://www.iaik.tu-graz.ac.at/");
      attachment.setDataHandler(new DataHandler(new URLDataSource(url)));
      attachment.setFileName("index.html");
      Multipart mp = new SMimeMultipart();
      mp.addBodyPart(mbp1);
      mp.addBodyPart(attachment);
      DataHandler multipart = new DataHandler(mp, mp.getContentType());

      Message msg;    // the message to send

      // 1. We send a plain message
      msg = createPlainMessage(session, multipart);
      System.out.println("sending plain message...");
	  Transport.send(msg);

      // 2. we send an implicitly signed message
      msg = createSignedMessage(session, multipart, true);
      System.out.println("sending implicitly signed message...");
	  Transport.send(msg);

      // 3. Now create encrypted messages with different content encryption algorithms and send them
      msg = createEncryptedMessage(session, AlgorithmID.rc2_CBC, 40);
      System.out.println("sending encrypted message [RC2/40]...");
	    Transport.send(msg);
      msg = createEncryptedMessage(session, AlgorithmID.rc2_CBC, 64);
      System.out.println("sending encrypted message [RC2/64]...");
	    Transport.send(msg);
      msg = createEncryptedMessage(session, AlgorithmID.rc2_CBC, 128);
      System.out.println("sending encrypted message [RC2/128]...");
	    Transport.send(msg);
      msg = createEncryptedMessage(session, AlgorithmID.des_CBC, 0);
      System.out.println("sending encrypted message [DES]...");
	    Transport.send(msg);
      msg = createEncryptedMessage(session, AlgorithmID.des_EDE3_CBC, 0);
      System.out.println("sending encrypted message [TripleDES]...");
	    Transport.send(msg);

      // 4. Now create a implicitly signed and encrypted message with attachment and send it
      msg = createSignedAndEncryptedMessage(session, multipart, true);
      System.out.println("sending implicitly signed and encrypted message [RC2/40]...");
	    Transport.send(msg);

      /*
      // 5. Now create a explicitly signed and encrypted message with attachment
      msg = createSignedAndEncryptedMessage(session, multipart, false);
      System.out.println("sending explicitly signed and encrypted message [RC2/40]...");
	    Transport.send(msg);
	  */
	 
	 //6. we send a cert request
     msg = createPKCS10Message(session);
     System.out.println("sending application/pkcs10 message...");
	 Transport.send(msg);
	 
	 
	  //7. we send an application/pkcs10 message where the request is in the second part
	  msg = createPKCS10MultiPartMessage(session);
	  System.out.println("sending message with pkcs10 part...");
	  Transport.send(msg);
  	
  	} catch (MessagingException mex) {
	    //something was not ok, so look at output
	    mex.printStackTrace();
	    Exception ex = null;
	    if ((ex = mex.getNextException()) != null) {
    		ex.printStackTrace();
	    }
  	}
    //now if we get here everything is ok
  	System.out.println("OK!");
  	System.in.read();
  }

  /**
   * creates a message
   * 
   * @param session
   * @exception MessagingException
   */
  public Message createMessage(Session session) throws MessagingException {
    MimeMessage msg = new MimeMessage(session);
    msg.setFrom(new InternetAddress(from));
	  msg.setRecipients(Message.RecipientType.TO, InternetAddress.parse(to, false));
		msg.setSentDate(new Date());
		
		return msg;
  }
  
  /**
   * creates a plain message
   * 
   * @param session
   * @param dataHandler
   * @exception MessagingException
   */
  public Message createPlainMessage(Session session, DataHandler dataHandler) throws MessagingException {
  
    Message msg = createMessage(session);
    msg.setSubject("IAIK-S/MIME: Plain");
    if (dataHandler != null)
      msg.setDataHandler(dataHandler);
    else
      msg.setText("This is a plain message!\nIt is neither signed nor encrypted!\n");      
    
		return msg;
  }

  /**
   * creates a signed and encrypted message
   * 
   * @param session
   * @param dataHandler
   * @param implicit
   * @exception MessagingException
   */
  public Message createSignedAndEncryptedMessage(Session session, DataHandler dataHandler, boolean implicit) 
      throws MessagingException {
        
    Message msg = createMessage(session);
    StringBuffer buf = new StringBuffer();
    if (implicit) {
      msg.setSubject("IAIK-S/MIME: Implicitly Signed and Encrypted");
    //  buf.append("This message is implicitly signed and encrypted!\n");
    //  buf.append("\n\n");
      buf.append("TEST");
    }
    else {
      msg.setSubject("IAIK-S/MIME: Explicitly Signed and Encrypted");
      buf.append("This message is explicitly signed and encrypted!\n");
      buf.append("\n\n");
    }

    SignedContent sc = new SignedContent(implicit);
    if (dataHandler != null)
      sc.setDataHandler(dataHandler);
    else
      sc.setText(buf.toString());
    
    sc.setCertificates(signerCertificates);
    try {
      sc.setSigner(signerPrivateKey, signerCertificate);
    } catch (SignatureException ex) {
      throw new MessagingException("Error signing content!", ex);
    } catch (InvalidKeyException ex) {
      throw new MessagingException("Invalid Key!", ex);
    }

    EncryptedContent ec = new EncryptedContent(sc);
    // encrypt for the recipient
    ec.addRecipient(recipientCertificate, AlgorithmID.rsaEncryption);
    // I want to be able to decrypt the message, too
    ec.addRecipient(encryptionCertOfSigner, AlgorithmID.rsaEncryption);
    // set the encryption algorithm
    ec.setEncryptionAlgorithm(AlgorithmID.rc2_CBC, 40);
    msg.setContent(ec, ec.getContentType());
    // let the EncryptedContent update some message headers
    ec.setHeaders(msg);
    
    return msg;
  }

  /**
   * creates a signed message
   * 
   * @param session
   * @param dataHandler
   * @param implicit
   * @exception MessagingException
   */
  public Message createSignedMessage(Session session, DataHandler dataHandler, boolean implicit)
      throws MessagingException {

    Message msg = createMessage(session);

    SignedContent sc = new SignedContent(implicit);
    StringBuffer buf = new StringBuffer();
    if (implicit) {
      msg.setSubject("IAIK-S/MIME: Implicitly Signed");
      buf.append("This message is implicitly signed!\n");
      buf.append("You need an S/MIME aware mail client to view this message.\n");
      buf.append("\n\n");
      buf.append("TEST");
    }
    else {
      msg.setSubject("IAIK-S/MIME: Explicitly Signed");
      buf.append("This message is explicitly signed!\n");
      buf.append("Every mail client can view this message.\n");
      buf.append("Non S/MIME mail clients will show the signature as attachment.\n");
      buf.append("\n\n");
    }
    
    if (dataHandler != null)
      sc.setDataHandler(dataHandler);
    else
      sc.setText(buf.toString());
      
    sc.setCertificates(signerCertificates);

    try {
      sc.setSigner(signerPrivateKey, signerCertificate);
    } catch (SignatureException ex) {
      throw new MessagingException("Error signing content!", ex);
    } catch (InvalidKeyException ex) {
      throw new MessagingException("Invalid Key!", ex);
    }
    
    msg.setContent(sc, sc.getContentType());
    return msg;
  }
  
  /**
   * creates an encrypted message
   * 
   * @param session
   * @param algorithm
   * @param keyLength
   * @exception MessagingException
   */
  public Message createEncryptedMessage(Session session, AlgorithmID algorithm, int keyLength)
      throws MessagingException {

    Message msg = createMessage(session);
    StringBuffer subject = new StringBuffer();
    subject.append("IAIK-S/MIME: Encrypted ["+algorithm.getName());
    if (keyLength > 0)
      subject.append("/"+keyLength);
    subject.append("]");
    msg.setSubject(subject.toString());

    EncryptedContent ec = new EncryptedContent();

    StringBuffer buf = new StringBuffer();
    buf.append("This is the encrypted content!\n");
    buf.append("Content encryption algorithm: "+algorithm.getName());
    buf.append("\n\n");
    
    ec.setText(buf.toString());
    // encrypt for the recipient
    ec.addRecipient(recipientCertificate, AlgorithmID.rsaEncryption);
    // I want to be able to decrypt the message, too
    ec.addRecipient(signerCertificate, AlgorithmID.rsaEncryption);
    ec.setEncryptionAlgorithm(algorithm, keyLength);

    msg.setContent(ec, ec.getContentType());
    //let the EncryptedContent update some message headers
    ec.setHeaders(msg);

    return msg;  
  }
  
  /**
   * creates a certificate only message
   * 
   * @param session
   * @exception MessagingException
   */
  public Message createCertsOnlyMessage(Session session)
      throws MessagingException {

    Message msg = createMessage(session);
    msg.setSubject("IAIK S/MIME: Certs-only message");
    //use new content types
    SMimeParameters.useNewContentTypes(true);
    SignedContent sc = new SignedContent(true, SignedContent.CERTS_ONLY);
    sc.setCertificates(signerCertificates);
    msg.setContent(sc, sc.getContentType());
    //set filename and attachment parameters
    sc.setHeaders(msg);
    
  
    return msg;
  }
  
  /**
   * creates a certificate only multipart message
   * 
   * @param session
   * @exception MessagingException
   */
  public Message createCertsOnlyMultiPartMessage(Session session) throws MessagingException {
  
      MimeBodyPart mbp1 = new MimeBodyPart();
	  mbp1.setText("This is a test where the certs-only message is included in the second part!\n\n");
	  
      MimeBodyPart attachment = new MimeBodyPart();
      //use new content types
      SMimeParameters.useNewContentTypes(true);
      SignedContent sc = new SignedContent(true, SignedContent.CERTS_ONLY);
      sc.setCertificates(signerCertificates);
      attachment.setContent(sc);
      Multipart mp = new MimeMultipart();
      mp.addBodyPart(mbp1);
      mp.addBodyPart(attachment);
           
      Message msg = createMessage(session);
      msg.setSubject("IAIK S/MIME: Certs-only message");
     
      msg.setContent(mp, mp.getContentType());
   
      return msg;
    
  }  
  
   /**
    * creates a certificate request message
    * 
    * @param session
    * @exception MessagingException
    */
   public Message createPKCS10Message(Session session)
      throws MessagingException {

    Message msg = createMessage(session);
    StringBuffer subject = new StringBuffer();
    subject.append("IAIK-S/MIME: Certificate Request");
    msg.setSubject(subject.toString());

    PKCS10Content pc = new PKCS10Content();
    CertificateRequest request = null;
    try {
       request = createCertificateRequest();
    } catch (PKCSException ex) {
       throw new MessagingException(ex.getMessage());  
    }  
    pc.setCertRequest(request);
    msg.setContent(pc, pc.getContentType());
    // let the PKCS10Content update some message headers
    pc.setHeaders(msg);
    
    return msg;  
  }
  
  
  /**
   * creates a certificate request
   * 
   * @exception iaik.pkcs.PKCSException
   */
  private CertificateRequest createCertificateRequest() throws PKCSException {
    try {
        Name subject = new Name();
	    subject.addRDN(ObjectID.commonName, firstName + " " + lastName);
	    subject.addRDN(ObjectID.emailAddress, from);
	    CertificateRequest certRequest;
	 
        certRequest = new CertificateRequest(signerCertificate.getPublicKey(), subject);
  	    certRequest.sign(AlgorithmID.sha1WithRSAEncryption, signerPrivateKey);
  	    certRequest.verify();
  	    return certRequest;
  	} catch (Exception ex) {
  	    throw new PKCSException("Cannot create cert request: " + ex.getMessage());   
  	}    
    
  }  
  
  /**
   * creates a certificate request multipart message
   * 
   * @param session
   * @exception MessagingException
   */
  public Message createPKCS10MultiPartMessage(Session session) throws MessagingException {
  
      MimeBodyPart mbp1 = new MimeBodyPart();
	  mbp1.setText("This is a test where the request message is included in the second part!\n\n");
	  // try to test an attachment
	  // this demo attaches our homepage
      MimeBodyPart attachment = new MimeBodyPart();
      //use new content types
      SMimeParameters.useNewContentTypes(true);
      PKCS10Content pc = new PKCS10Content();
      CertificateRequest request = null;
      try {
         request = createCertificateRequest();
      } catch (PKCSException ex) {
         throw new MessagingException(ex.getMessage());  
      }  
      pc.setCertRequest(request);
      DataHandler pkcs10Handler = new DataHandler(pc, pc.getContentType());
      attachment.setDataHandler(pkcs10Handler);
      attachment.setDisposition("attachment");
      attachment.setFileName("smime.p10");
      Multipart mp = new MimeMultipart();
      mp.addBodyPart(mbp1);
      mp.addBodyPart(attachment);
           
      Message msg = createMessage(session);
      msg.setSubject("IAIK-S/MIME: Certificate Request");
     
      msg.setContent(mp, mp.getContentType());
   
      return msg;
    
  } 
  
  

  public static void main(String[] argv) throws Exception {

    (new SMimeSend()).start();
   	iaik.utils.Util.waitKey();
   	return;
  }
  
  private String[] getPKIadress()
  {
    String[] info = new String[2];
    try{
        String filename = properties.getString(section,"inifile");
        IniFile ini = new IniFile(filename);
        String newline = ini.getString("Entrust Settings", "Server", "");
        if ( newline.equals("") )
        {
            System.out.println("The Server address not found.");
        }
        else
        {
                //we found the entry for Server address
                System.out.println("we found the entry for Server address");
                info[0] = newline.substring(0,newline.indexOf("+"));
                info[1] = newline.substring((newline.indexOf("+")+1));
                System.out.println("Server: " + info[0]);
                System.out.println("Port: " + info[1]);
        }
    }
    catch(FileNotFoundException ex){
        System.out.println("Could not load entrust.ini file.");
    }
    return info;
  }
}