//===========================================================================
//
// Copyright (c)  1997-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.smimev3;

import iaik.asn1.ObjectID;
import iaik.asn1.structures.AlgorithmID;
import iaik.asn1.structures.Name;
import iaik.cms.CompressionProvider;
import iaik.pkcs.PKCSException;
import iaik.pkcs.pkcs10.CertificateRequest;
import iaik.smime.CompressedContent;
import iaik.smime.EncryptedContent;
import iaik.smime.PKCS10Content;
import iaik.smime.SMimeBodyPart;
import iaik.smime.SMimeContentType;
import iaik.smime.SMimeMultipart;
import iaik.smime.SMimeParameters;
import iaik.smime.SignedContent;
import iaik.x509.X509Certificate;

import java.io.FileOutputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URL;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.util.Date;
import java.util.Properties;

import javax.activation.CommandMap;
import javax.activation.DataHandler;
import javax.activation.MailcapCommandMap;
import javax.activation.URLDataSource;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Part;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.exceptions.UserNotLoggedInException;
import com.entrust.toolkit.util.IniFile;
import com.entrust.toolkit.util.ManagerTransport;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.directory.JNDIDirectory;

/**
 * This class demonstrates the usage of the IAIK S/MIME implementation. To run
 * this demo you also need the packages:
 * <ul>
 * <li>mail.jar: Get it from <a
 * href="http://www.javasoft.com/products/javamail/">Sun </a>.
 * <li>activation.jar: Get it from <a
 * href="http://java.sun.com/beans/glasgow/jaf.html">Sun </a>.
 * <li>OPTIONAL: pop3.jar: If you need POP3 access. Get it from <a
 * href="http://www.javasoft.com/products/javamail/">here </a>.
 * </ul>
 *  
 */
public class SMimeSend {
	/**
	 * Array of signer's verification certificate and CA certificate
	 */
	X509Certificate[] signerCertificates = null;

	/**
	 * encryption certificate of the recipient
	 */
	X509Certificate recipientCertificate = null;

	/**
	 * verification certificate of the signer
	 */
	X509Certificate signerCertificate = null;

	/**
	 * encryption certificate of the signer
	 */
	X509Certificate encryptionCertOfSigner = null;

	/**
	 * private key of the signer
	 */
	PrivateKey signerPrivateKey = null;

	IniFile properties = null;

	String section = null;

	String attrToFind = null;

	String searchbase = null;

	String searchexpr = null;

	String to = null;

	String from = null;

	String host = null;

	String Profile = null;

	String password = null;

	String pki = null;

	String port = null;

	String firstName = null;

	String lastName = null;

	String mailcapFile = null;

	/**
	 * Creates a new SMimeSend object
	 */
	public SMimeSend() throws Exception {

		//get the properties from example.ini
		try {
			properties = new IniFile("data/smime/smime_example.ini");
		} catch (FileNotFoundException ex) {
			System.out
					.println("Could not found the file data/smime/smime_example.ini: "
							+ ex.getMessage());
		}
		section = new String("SMimeSend");
		//Search criteria for Recipient
		attrToFind = properties.getString(section, "attrToFind");
		//to edit begin
		searchbase = properties.getString(section, "searchbase");
		searchexpr = properties.getString(section, "searchexpr");
		to = properties.getString(section, "to");
		from = properties.getString(section, "from");
		host = properties.getString(section, "host");
		//Profile of Sender
		Profile = properties.getString(section, "Profile");
		password = properties.getString(section, "password");
		//to edit end
		// path to mailcap file, defaults to "mailcap"
		mailcapFile = properties.getString(section, "mailcap", "mailcap");

		//now we read the profile of the sender into an Inputstream
		FilenameProfileReader fis = null;
		try {
			fis = new FilenameProfileReader(Profile);
		} catch (FileNotFoundException e) {
			System.out.println("Profile not found!");
			throw e;
		}

		User sender = new User();
		try {
			//connect to the JNDIDirectory ->get information from the
			// entrust.ini File
			String[] info = getPKIadress();
			pki = info[0];
			port = info[1];
			if (info[0] != null && info[1] != null) {
				JNDIDirectory dir = new JNDIDirectory(pki, (new Integer(port))
						.intValue());
				ManagerTransport emt = new ManagerTransport(pki, 829);

				sender.setConnections(dir, emt);
				sender.login(fis, new SecureStringBuffer(password));

				//get the recipient certificate from the directory
				byte[][] result = dir
						.Search(searchbase, searchexpr, attrToFind);
				recipientCertificate = new X509Certificate(result[0]);
			} else {
				String certificateFile = properties.getString(section,
						"recipientCertificate");
				System.out.println("Working offline, using certificate file "
						+ certificateFile + " as recipient");
				recipientCertificate = new X509Certificate(new FileInputStream(
						certificateFile));
				sender.login(fis, new SecureStringBuffer(password));
			}
		} catch (Exception e) {
			System.out.println("Login failed: " + e.getMessage());
			throw e;
		}
		//lets check if recipientCertificate is a valid certificate, i.e.,
		// there is a valid certificate path from recipientCertificate
		//to the root of trust.
		try {
			sender.validate(recipientCertificate);
			System.out.println("Recipient's Certificate trusted!");
		} catch (Exception ex) {
			System.out.println("Recipient's Certificate not trusted!");
		}

		X509Certificate[] certs = new X509Certificate[2];
		try {
			certs[0] = sender.getVerificationCertificate();
			certs[1] = sender.getCaCertificate();

			signerCertificates = certs;
			signerPrivateKey = sender.getSigningKey();
			signerCertificate = sender.getVerificationCertificate();

			// recipient = signer for this test
			encryptionCertOfSigner = sender.getEncryptionCertificate();
			System.out.println("Algorithm :"
					+ sender.getEncryptionCertificate().getPublicKey()
							.getAlgorithm());
		} catch (UserNotLoggedInException e) {
			System.out.print(e.getMessage());
			System.out.print("User not logged in!");
		}

		for (int i = 0; i < signerCertificates.length; i++)
			System.out.println(signerCertificates[i].toString(true));
		System.out.println(recipientCertificate.toString(true));
		System.out.println(encryptionCertOfSigner.toString(true));
	}

	public void start() throws IOException {
		// register content data handlers for S/MIME types
		MailcapCommandMap mc = new MailcapCommandMap(mailcapFile);
		CommandMap.setDefaultCommandMap(mc);

		// create some properties and get the default Session
		Properties props = new Properties();
		props.put("mail.smtp.host", host);
		//props.put("mail.debug", "true");
		Session session = Session.getDefaultInstance(props, null);
		Session session2 = Session.getDefaultInstance(props, null);
		//session.setDebug(true);

		try {
			// Create a demo Multipart
			MimeBodyPart mbp1 = new SMimeBodyPart();
			mbp1.setText("This is a Test of the IAIK S/MIME implementation!\n\n");
			// try to test an attachment
			// this demo attaches our homepage
			MimeBodyPart attachment = new SMimeBodyPart();
			Multipart mp = new SMimeMultipart();
			mp.addBodyPart(mbp1);
			DataHandler multipart = new DataHandler(mp, mp.getContentType());

			Message msg; // the message to send
			Message msg2; // the message to send

			// 1. This is a plain message
			msg = createPlainMessage(session, multipart);
			System.out.println("sending plain message...");
			Transport.send(msg);
			//Compress the message
			msg2 = addCompressionEnvelope(msg, session,CompressionProvider.zlib_compress);
			System.out.println("sending plain message compressed...");
			Transport.send(msg2);

			// 2. This is an explicitly signed message
			msg = createSignedMessage(session, multipart, false);
			System.out.println("sending explicitly signed message...");
			Transport.send(msg);
			msg2 = addCompressionEnvelope(msg, session,CompressionProvider.zlib_compress);
			System.out.println("sending  explicitly signed message compressed...");
			Transport.send(msg2);
			
			// 3. This is an implicitly signed message
			msg = createSignedMessage(session, multipart, true);
			System.out.println("sending implicitly signed message...");
			Transport.send(msg);
			// Compress the message
			msg2 = addCompressionEnvelope(msg, session,CompressionProvider.zlib_compress);
			System.out.println("sending implicitly signed message compressed...");
			Transport.send(msg2);
			
			// Encrypt the message
			// 4. Now create encrypted messages with different content encryption algorithms
			msg = createEncryptedMessage(session, AlgorithmID.rc2_CBC, 40);
			System.out.println("sending encrypted message [RC2/40]...");
			Transport.send(msg);
			// Compress the message
			msg2 = addCompressionEnvelope(msg, session,CompressionProvider.zlib_compress);
			System.out.println("sending encrypted message [RC2/40] compressed...");
			Transport.send(msg2);

			msg = createEncryptedMessage(session, AlgorithmID.rc2_CBC, 64);
			System.out.println("sending encrypted message [RC2/64]...");
			Transport.send(msg);
			// Compress the message
			msg2 = addCompressionEnvelope(msg, session,CompressionProvider.zlib_compress);
			System.out.println("sending encrypted message [RC2/64] compressed...");
			Transport.send(msg2);

			msg = createEncryptedMessage(session, AlgorithmID.rc2_CBC, 128);
			System.out.println("sending encrypted message [RC2/128]...");
			Transport.send(msg);
			// Compress the message
			msg2 = addCompressionEnvelope(msg, session,CompressionProvider.zlib_compress);
			System.out.println("sending encrypted message [RC2/128] compressed...");
			Transport.send(msg2);

			msg = createEncryptedMessage(session, AlgorithmID.des_EDE3_CBC, 0);
			System.out.println("sending encrypted message [TripleDES]...");
			Transport.send(msg);
			// Compress the message
			msg2 = addCompressionEnvelope(msg, session,CompressionProvider.zlib_compress);
			System.out.println("sending encrypted message [TripleDES] compressed...");
			Transport.send(msg2);

			// 5. Now create a implicitly signed and encrypted message with attachment
			msg = createSignedAndEncryptedMessage(session, multipart, true);
			System.out.println("sending implicitly signed and encrypted message [RC2/40]...");
			Transport.send(msg);
			// Compress the message
			msg2 = addCompressionEnvelope(msg, session,CompressionProvider.zlib_compress);
			System.out.println("sending implicitly signed and encrypted message [RC2/40] compressed...");
			Transport.send(msg2);

			// 6. Now create a explicitly signed and encrypted message with attachment
			msg = createSignedAndEncryptedMessage(session, multipart, false);
			System.out.println("sending explicitly signed and encrypted message [RC2/40]...");
			Transport.send(msg);
			// Compress the message
			msg2 = addCompressionEnvelope(msg, session,CompressionProvider.zlib_compress);
			System.out.println("sending explicitly signed and encrypted message [RC2/40] compressed...");
			Transport.send(msg2);
			
			// 7. certs only message
			msg = createCertsOnlyMessage(session);
			System.out.println("sending certs-only message");
			Transport.send(msg);
			// Compress the message
			msg2 = addCompressionEnvelope(msg, session,CompressionProvider.zlib_compress);
			System.out.println("sending certs-only message compressed...");
			Transport.send(msg2);

			// 8. second certs only message
			msg = createCertsOnlyMultiPartMessage(session);
			System.out.println("sending message with certs-only part");
			Transport.send(msg);
            // Compress the message
			msg2 = addCompressionEnvelope(msg, session,CompressionProvider.zlib_compress);
			System.out.println("sending message with certs-only part compressed...");
			Transport.send(msg2);

			// 9. sending cert request
			msg = createPKCS10Message(session);
			System.out.println("sending application/pkcs10 message...");
			Transport.send(msg);
			// Compress the message
			msg2 = addCompressionEnvelope(msg, session,CompressionProvider.zlib_compress);
			System.out.println("sending application/pkcs10 message compressed...");
		 	Transport.send(msg2);

			// 10. ending application/pkcs10 message where the request is in the second part
			msg = createPKCS10MultiPartMessage(session);
			System.out.println("sending message with pkcs10 part...");
			Transport.send(msg);
            // Compress the message
			msg2 = addCompressionEnvelope(msg, session,CompressionProvider.zlib_compress);
			System.out.println("sending message with pkcs10 part compressed...");
			Transport.send(msg2);

			//Now show some additional SMimeCompression usage
			msg = createCompressedMessage(session, multipart,CompressionProvider.zlib_compress);
			System.out.println("sending compressed message... ");
			Transport.send(msg);

			// 11. Demonstrate usage of a Null Compression Algorithm in a Signed
			// Compressed Message
			msg2 = createSignedCompressedMessage(session, multipart, true,NullCompression.alg_null);
			System.out.println("sending signed and compressed data with sample alg_null compression");
			Transport.send(msg2);

			// 12. Demonstrate usage of the standard zlib_compression in a
			// signed and Compressed message
			msg2 = createSignedCompressedMessage(session, multipart, true,CompressionProvider.zlib_compress);
			System.out.println("sending signed and compressed data with zlib_compression compression");
			Transport.send(msg2);
			
			//An alternative method of wrapping Signed, encrypted and compressed messages is shown here
			SignedContent sc = createSignedContent(session, multipart, false);
			EncryptedContent ec = createEncryptedContent(session,CompressionProvider.des_EDE3_CBC, 0);
			CompressedContent cc = createCompressedContent(session,multipart, CompressionProvider.zlib_compress);

			// 13. Send a Signed and compressed message
			Object[] o = { cc, sc,  };
			Message message = wrapContentTypes(o, session);
			System.out.println("sending signed and compressed message using different wrapped example...");
			Transport.send(message);

			// 14. Send an encrypted and compressed message. Note that compressing encrypted content is not
			// really beneficial since encrypted content is inherently random. This is only for
			// demonstration purposes
			Object[] o2 = { cc, ec };
			message = wrapContentTypes(o2, session);
			System.out.println("sending signed and encrypted message using different wrapped example...");
			Transport.send(message);

			// 15. Signed and Encrypt a message
			// Note:  New signature needs to be generated each time, or signature validation will
			// fail because wrapped content changes each time.
			sc = createSignedContent(session, multipart, false);
			Object[] o3 = { ec, sc };
			message = wrapContentTypes(o3, session);
			System.out.println("sending signed and encrypted message using different wrapped example...");
			Transport.send(message);
			
			// 16. Send a signed encrypted and compressed message.
			sc = createSignedContent(session, multipart, false);
			Object[] o4 = { cc, ec, sc };
			message = wrapContentTypes(o4, session);
			System.out.println("sending signed, encrypted and compressed message using different wrapped example...");
			Transport.send(message);

			// 17. Send an encrypted, signed and compressed message.
			sc = createSignedContent(session, multipart, false);
			Object[] o5 = { cc, sc, ec };
			message = wrapContentTypes(o5, session);
			System.out.println("sending encrypted, signed and compressed message using different wrapped example...");
			Transport.send(message);

			// 18. Send a Compressed, Encrypted, Compressed, Signed, Compressed
			// message. Content types can be arbitrarily nested according to RFC 3851
			CompressedContent cc2 = createCompressedContent(session, null,CompressionProvider.zlib_compress);
			CompressedContent cc3 = createCompressedContent(session, null,CompressionProvider.zlib_compress);
			sc = createSignedContent(session, multipart, false);
			Object[] o6 = { cc, sc, cc2, ec, cc3 };
			message = wrapContentTypes(o6, session);
			System.out.println("sending compressed, encrypted, compressed, signed and compressed message using different wrapped example...");
			Transport.send(message);

		} catch (MessagingException mex) {
			mex.printStackTrace();
			Exception ex = null;
			if ((ex = mex.getNextException()) != null) {
				ex.printStackTrace();
			}
		}

		catch (UserNotLoggedInException uex) {
			uex.printStackTrace();
		}

		catch (Exception e) {
			e.printStackTrace();
		}

		System.out.println("OK!");
		System.in.read();
	}

	public SignedContent createSignedContent(Session session,
			DataHandler dataHandler, boolean implicit)
			throws MessagingException {

		// Message msg = createMessage(session);

		SignedContent sc = new SignedContent(implicit);
		StringBuffer buf = new StringBuffer();
		if (implicit) {
			//msg.setSubject("IAIK-S/MIME: Implicitly Signed");
			buf.append("This message is implicitly signed!\n");
			buf.append("You need an S/MIME aware mail client to view this message.\n");
			buf.append("\n\n");
			buf.append("TEST");
		} else {
			//msg.setSubject("IAIK-S/MIME: Explicitly Signed");
			buf.append("This message is explicitly signed!\n");
			buf.append("Every mail client can view this message.\n");
			buf.append("Non S/MIME mail clients will show the signature as attachment.\n");
			buf.append("\n\n");
		}

		if (dataHandler != null)
			sc.setDataHandler(dataHandler);
		else
			sc.setText(buf.toString());

		sc.setCertificates(signerCertificates);
		try {
			sc.addSigner(signerPrivateKey, signerCertificate);
		} catch (NoSuchAlgorithmException ex) {
			throw new MessagingException("Algorithm not supported: "
					+ ex.getMessage(), ex);

		} catch (InvalidKeyException ex) {
			throw new MessagingException("Invalid key specified for signing: "
					+ ex.getMessage(), ex);
		}

		return sc;
	}

	public EncryptedContent createEncryptedContent(Session session,
			AlgorithmID algorithm, int keyLength) throws MessagingException,
			UserNotLoggedInException {

		EncryptedContent ec = new EncryptedContent();

		StringBuffer buf = new StringBuffer();
		buf.append("This is the encrypted content!\n");
		buf.append("Content encryption algorithm: " + algorithm.getName());
		buf.append("\n\n");

		ec.setText(buf.toString());
		// encrypt for the recipient
		ec.addRecipient(recipientCertificate, AlgorithmID.rsaEncryption);
		// I want to be able to decrypt the message, too
		ec.setEncryptionAlgorithm(algorithm, keyLength);
		return ec;
	}

	/**
	 * Creates a compressed message.
	 * 
	 * @param session
	 *            the mail session
	 * @param dataHandler
	 *            the datahandler supplying the content to be compressed
	 * @param algorithm
	 *            the compression algorithm to be used
	 * 
	 * @return the compressed message
	 * 
	 * @exception MessagingException
	 *                if an error occurs when creating the message
	 */
	public CompressedContent createCompressedContent(Session session,
			DataHandler dataHandler, AlgorithmID algorithm)
			throws MessagingException {

		String subject = "IAIK-S/MIME: Compressed [" + algorithm.getName()
				+ "]";
		CompressedContent compressedContent = new CompressedContent();

		if (dataHandler == null) {
			StringBuffer buf = new StringBuffer();
			buf.append("This is the compressed content!  Created by John Gray\n");
			buf.append("Compression algorithm: " + algorithm.getName());
			buf.append("\n\n");
			compressedContent.setText(buf.toString());
		} else {
			compressedContent.setDataHandler(dataHandler);
		}

		try {
			compressedContent.setCompressionAlgorithm(algorithm);
		} catch (NoSuchAlgorithmException ex) {
			throw new MessagingException(
					"Compression algorithm not supported: " + ex.getMessage());
		}

		return compressedContent;
	}

	public Message createMessage(Session session) throws MessagingException {
		MimeMessage msg = new MimeMessage(session);
		msg.setFrom(new InternetAddress(from));
		msg.setRecipients(Message.RecipientType.TO, InternetAddress.parse(to,false));
		msg.setSentDate(new Date());

		return msg;
	}

	public Message createPlainMessage(Session session, DataHandler dataHandler)
			throws MessagingException {
		Message msg = createMessage(session);
		msg.setSubject("IAIK-S/MIME: Plain");
		if (dataHandler != null)
			msg.setDataHandler(dataHandler);
		else
			msg.setText("This is a plain message!\nIt is whether signed nor encrypted!\n");

		return msg;
	}

	public Message createSignedAndEncryptedMessage(Session session,
			DataHandler dataHandler, boolean implicit)
			throws MessagingException {
		Message msg = createMessage(session);
		StringBuffer buf = new StringBuffer();
		if (implicit) {
			msg.setSubject("IAIK-S/MIME: Implicitly Signed and Encrypted");
			buf.append("This message is implicitly signed and encrypted!\n");
			buf.append("\n\n");
			buf.append("TEST");
		} else {
			msg.setSubject("IAIK-S/MIME: Explicitly Signed and Encrypted");
			buf.append("This message is explicitly signed and encrypted!\n");
			buf.append("\n\n");
		}

		SignedContent sc = new SignedContent(implicit);
		if (dataHandler != null)
			sc.setDataHandler(dataHandler);
		else
			sc.setText(buf.toString());

		sc.setCertificates(signerCertificates);
		try {
			sc.addSigner(signerPrivateKey, signerCertificate);
		} catch (NoSuchAlgorithmException ex) {
			throw new MessagingException("Algorithm not supported: "
					+ ex.getMessage(), ex);
		} catch (InvalidKeyException ex) {
			throw new MessagingException("Invalid key specified for signing: "
					+ ex.getMessage(), ex);
		}

		EncryptedContent ec = new EncryptedContent(sc);
		// encrypt for the recipient
		ec.addRecipient(recipientCertificate, AlgorithmID.rsaEncryption);
		// I want to be able to decrypt the message, too
		ec.addRecipient(encryptionCertOfSigner, AlgorithmID.rsaEncryption);
		// set the encryption algorithm
		ec.setEncryptionAlgorithm(AlgorithmID.rc2_CBC, 40);
		msg.setContent(ec, ec.getContentType());
		// let the EncryptedContent update some message headers
		ec.setHeaders(msg);

		return msg;
	}

	public Message createEncryptedAndSignedMessage(Session session,
			DataHandler dataHandler, boolean implicit)
			throws MessagingException {
		Message msg = createMessage(session);
		StringBuffer buf = new StringBuffer();
		if (implicit) {
			msg.setSubject("IAIK-S/MIME: Implicitly Encrypted and Signed Message");
			buf.append("This message is implicitly signed and encrypted!\n");
			buf.append("\n\n");
			buf.append("TEST");
		} else {
			msg.setSubject("IAIK-S/MIME: Explicitly Encrypted and Signed");
			buf.append("This message is explicitly Encrypted and Signed!\n");
			buf.append("\n\n");
		}

		EncryptedContent ec = new EncryptedContent();
		// encrypt for the recipient
		ec.addRecipient(recipientCertificate, AlgorithmID.rsaEncryption);
		// I want to be able to decrypt the message, too
		ec.addRecipient(encryptionCertOfSigner, AlgorithmID.rsaEncryption);
		// set the encryption algorithm
		ec.setEncryptionAlgorithm(AlgorithmID.rc2_CBC, 40);

		if (dataHandler != null)
			ec.setDataHandler(dataHandler);
		else
			ec.setText(buf.toString());

		// Wrap Encrypted Content with Signed Content
		SignedContent sc = new SignedContent(ec, implicit);

		sc.setCertificates(signerCertificates);
		try {
			sc.addSigner(signerPrivateKey, signerCertificate);
		} catch (NoSuchAlgorithmException ex) {
			throw new MessagingException("Algorithm not supported: "
					+ ex.getMessage(), ex);
		} catch (InvalidKeyException ex) {
			throw new MessagingException("Invalid key specified for signing: "
					+ ex.getMessage(), ex);
		}

		msg.setContent(sc, sc.getContentType());
		// let the EncryptedContent update some message headers
		ec.setHeaders(msg);

		return msg;
	}

	public Message createSignedMessage(Session session,
			DataHandler dataHandler, boolean implicit)
			throws MessagingException {

		Message msg = createMessage(session);

		SignedContent sc = new SignedContent(implicit);
		StringBuffer buf = new StringBuffer();
		if (implicit) {
			msg.setSubject("IAIK-S/MIME: Implicitly Signed");
			buf.append("This message is implicitly signed!\n");
			buf.append("You need an S/MIME aware mail client to view this message.\n");
			buf.append("\n\n");
			buf.append("TEST");
		} else {
			msg.setSubject("IAIK-S/MIME: Explicitly Signed");
			buf.append("This message is explicitly signed!\n");
			buf.append("Every mail client can view this message.\n");
			buf.append("Non S/MIME mail clients will show the signature as attachment.\n");
			buf.append("\n\n");
		}

		if (dataHandler != null)
			sc.setDataHandler(dataHandler);
		else
			sc.setText(buf.toString());

		sc.setCertificates(signerCertificates);

		try {
			sc.addSigner(signerPrivateKey, signerCertificate); // by default,
			// uses sha and rsa
			//sc.addSigner(signerPrivateKey, signerCertificate,
			// AlgorithmID.sha1, AlgorithmID.dsa);
		} catch (NoSuchAlgorithmException ex) {
			throw new MessagingException("Algorithm not supported: "
					+ ex.getMessage(), ex);
		} catch (InvalidKeyException ex) {
			throw new MessagingException("Invalid key specified for signing: "
					+ ex.getMessage(), ex);
		}

		msg.setContent(sc, sc.getContentType());
		return msg;
	}

	public Message createSignedCompressedMessage(Session session,
			DataHandler dataHandler, boolean implicit, AlgorithmID algorithm)
			throws MessagingException {

		Message msg = createMessage(session);

		SignedContent sc = new SignedContent(implicit);
		StringBuffer buf = new StringBuffer();
		if (implicit) {
			msg.setSubject("IAIK-S/MIME: Implicitly Signed");
			buf.append("This message is implicitly signed!\n");
			buf.append("You need an S/MIME aware mail client to view this message.\n");
			buf.append("\n\n");
			buf.append("TEST");
		} else {
			msg.setSubject("IAIK-S/MIME: Explicitly Signed");
			buf.append("This message is explicitly signed!\n");
			buf.append("Every mail client can view this message.\n");
			buf.append("Non S/MIME mail clients will show the signature as attachment.\n");
			buf.append("\n\n");
		}

		if (dataHandler != null)
			sc.setDataHandler(dataHandler);
		else
			sc.setText(buf.toString());

		sc.setCertificates(signerCertificates);
		CompressedContent cc = new CompressedContent(sc);
		NullCompression nullprovider = new NullCompression();
		cc.setCompressionProvider(nullprovider);
		try {
			sc.addSigner(signerPrivateKey, signerCertificate); // by default,
			cc.setCompressionAlgorithm(algorithm);
		} catch (NoSuchAlgorithmException ex) {
			throw new MessagingException("Algorithm not supported: "
					+ ex.getMessage(), ex);
		} catch (InvalidKeyException ex) {
			throw new MessagingException("Invalid key specified for signing: "
					+ ex.getMessage(), ex);
		}

		msg.setContent(cc, cc.getContentType());
		return msg;
	}

	public Message createEncryptedMessage(Session session,
			AlgorithmID algorithm, int keyLength) throws MessagingException {

		Message msg = createMessage(session);
		StringBuffer subject = new StringBuffer();
		subject.append("IAIK-S/MIME: Encrypted [" + algorithm.getName());
		if (keyLength > 0)
			subject.append("/" + keyLength);
		subject.append("]");
		msg.setSubject(subject.toString());

		EncryptedContent ec = new EncryptedContent();

		StringBuffer buf = new StringBuffer();
		buf.append("This is the encrypted content!\n");
		buf.append("Content encryption algorithm: " + algorithm.getName());
		buf.append("\n\n");

		ec.setText(buf.toString());
		// encrypt for the recipient
		ec.addRecipient(recipientCertificate, AlgorithmID.rsaEncryption);
		// I want to be able to decrypt the message, too
		ec.addRecipient(encryptionCertOfSigner, AlgorithmID.rsaEncryption);
		ec.setEncryptionAlgorithm(algorithm, keyLength);

		msg.setContent(ec, ec.getContentType());
		// let the EncryptedContent update some message headers
		ec.setHeaders(msg);

		return msg;
	}
	
	public Message createCertsOnlyMessage(Session session)
			throws MessagingException {
		Message msg = createMessage(session);
		msg.setSubject("IAIK S/MIME: Certs-only message");
		//use new content types
		SMimeParameters.useNewContentTypes(true);
		SignedContent sc = new SignedContent(true, SignedContent.CERTS_ONLY);
		sc.setCertificates(signerCertificates);
		msg.setContent(sc, sc.getContentType());
		//set filename and attachment parameters
		sc.setHeaders(msg);

		return msg;
	}

	public Message createCertsOnlyMultiPartMessage(Session session)
			throws MessagingException {
		MimeBodyPart mbp1 = new MimeBodyPart();
		mbp1
				.setText("This is a test where the certs-only message is included in the second part!\n\n");

		MimeBodyPart attachment = new MimeBodyPart();
		//use new content types
		SMimeParameters.useNewContentTypes(true);
		SignedContent sc = new SignedContent(true, SignedContent.CERTS_ONLY);
		sc.setCertificates(signerCertificates);
		attachment.setContent(sc);
		Multipart mp = new MimeMultipart();
		mp.addBodyPart(mbp1);
		mp.addBodyPart(attachment);

		Message msg = createMessage(session);
		msg.setSubject("IAIK S/MIME: Certs-only message");

		msg.setContent(mp, mp.getContentType());

		return msg;
	}

	public Message createPKCS10Message(Session session)
			throws MessagingException {
		Message msg = createMessage(session);
		StringBuffer subject = new StringBuffer();
		subject.append("IAIK-S/MIME: Certificate Request");
		msg.setSubject(subject.toString());

		PKCS10Content pc = new PKCS10Content();
		CertificateRequest request = null;
		try {
			request = createCertificateRequest();
		} catch (PKCSException ex) {
			throw new MessagingException(ex.getMessage());
		}
		pc.setCertRequest(request);
		msg.setContent(pc, pc.getContentType());
		// let the PKCS10Content update some message headers
		pc.setHeaders(msg);

		return msg;
	}

	private CertificateRequest createCertificateRequest() throws PKCSException {
		try {
			Name subject = new Name();
			subject.addRDN(ObjectID.commonName, firstName + " " + lastName);
			subject.addRDN(ObjectID.emailAddress, from);
			CertificateRequest certRequest;

			certRequest = new CertificateRequest(signerCertificate
					.getPublicKey(), subject);
			certRequest.sign(AlgorithmID.sha1WithRSAEncryption,
					signerPrivateKey);
			certRequest.verify();
			return certRequest;
		} catch (Exception ex) {
			throw new PKCSException("Cannot create cert request: "
					+ ex.getMessage());
		}
	}

	public Message createPKCS10MultiPartMessage(Session session)
			throws MessagingException {
		MimeBodyPart mbp1 = new MimeBodyPart();
		mbp1
				.setText("This is a test where the request message is included in the second part!\n\n");
		// try to test an attachment
		// this demo attaches our homepage
		MimeBodyPart attachment = new MimeBodyPart();
		//use new content types
		SMimeParameters.useNewContentTypes(true);
		PKCS10Content pc = new PKCS10Content();
		CertificateRequest request = null;
		try {
			request = createCertificateRequest();
		} catch (PKCSException ex) {
			throw new MessagingException(ex.getMessage());
		}
		pc.setCertRequest(request);
		DataHandler pkcs10Handler = new DataHandler(pc, pc.getContentType());
		attachment.setDataHandler(pkcs10Handler);
		attachment.setDisposition("attachment");
		attachment.setFileName("smime.p10");
		Multipart mp = new MimeMultipart();
		mp.addBodyPart(mbp1);
		mp.addBodyPart(attachment);

		Message msg = createMessage(session);
		msg.setSubject("IAIK-S/MIME: Certificate Request");

		msg.setContent(mp, mp.getContentType());

		return msg;
	}

	public Message wrapMessage(SMimeContentType obj, Session session)
			throws MessagingException {
		Message msg = createMessage(session);
		msg.setSubject("SMime Message!  May be signed, encrypted and compressed in any order or combination!" +
				       " This is a " +obj.getContentType() + "message!");
		msg.setContent(obj, obj.getContentType());
		return msg;
	}

	public Message wrapContentTypes(Object[] contentTypes, Session session)
			throws MessagingException {

		// Loop through given Content Types, Nesting the Content in each
		for (int i = 0; i < contentTypes.length; i++) {

			if (contentTypes[i] instanceof SignedContent) {
				SignedContent sc = (SignedContent) contentTypes[i];
				//System.out.println("SignedContent ContentType: "
				//		+ sc.getContentType() + "\n\n");
				if (i != contentTypes.length - 1) {
					sc.setContent(contentTypes[i + 1],
							((SMimeContentType) contentTypes[i + 1])
									.getContentType());
				}
				// If reusing same object infinite loop can occur if loop is
				// created between the datahandlers
				else {
					StringBuffer buf = new StringBuffer();
					buf.append("This Content was reset by the SMimeSend Sample!!\n");
					buf.append("Signer Infos for signed content: "
							+ sc.getSignerInfos());
					buf.append("\n\n");
					sc.setText(buf.toString());
				}
			}

			if (contentTypes[i] instanceof CompressedContent) {
				CompressedContent cc = (CompressedContent) contentTypes[i];
				//System.out.println("CompressedContent ContentType: "
				//		+ cc.getContentType() + "\n\n");
				if (i != contentTypes.length - 1) {
					cc.setContent(contentTypes[i + 1],
							((SMimeContentType) contentTypes[i + 1])
									.getContentType());
				}

				//If reusing same object infinite loop can occur if loop is
				// created between the datahandlers
				else {
					StringBuffer buf = new StringBuffer();
					buf.append("This Content was reset by the SMimeSend Sample!\n");
					buf.append("Content compression algorithm: "
							+ cc.getCompressionAlgorithm());
					buf.append("\n\n");
					cc.setText(buf.toString());
				}
			}

			if (contentTypes[i] instanceof EncryptedContent) {
				EncryptedContent ec = (EncryptedContent) contentTypes[i];
				//System.out.println("EncryptedContent ContentType: "
				//		+ ec.getContentType() + "\n\n");
				if (i != contentTypes.length - 1) {
					ec.setContent(contentTypes[i + 1],
							((SMimeContentType) contentTypes[i + 1])
									.getContentType());
                //In case objects are re-used, set the datahandler of the final Layer
		        //to text else an infinite loop which will result in a stack overflow
			    //will be created.
				} else {
					StringBuffer buf = new StringBuffer();
					buf.append("This Content was reset by the SMimeSend Sample!\n");
					buf.append("Content encryption algorithm: "
							+ ec.getEncryptionAlgorithm());
					buf.append("\n\n");
					ec.setText(buf.toString());
				}
			}
		}
		
		// Wrap the message in a final Mime layer....
		return wrapMessage((SMimeContentType) contentTypes[0], session);
	}

	/**
	 * Creates a compressed message.
	 * 
	 * @param session
	 *            the mail session
	 * @param dataHandler
	 *            the datahandler supplying the content to be compressed
	 * @param algorithm
	 *            the compression algorithm to be used
	 * 
	 * @return the compressed message
	 * 
	 * @exception MessagingException
	 *                if an error occurs when creating the message
	 */
	public Message createCompressedMessage(Session session,
			DataHandler dataHandler, AlgorithmID algorithm)
			throws MessagingException {

		String subject = "IAIK-S/MIME: Compressed [" + algorithm.getName()
				+ "]";
		//  Message msg = createMessage(session, subject.toString());

		Message msg = createMessage(session);
		msg.setSubject(subject);

		CompressedContent compressedContent = new CompressedContent();

		if (dataHandler == null) {
			StringBuffer buf = new StringBuffer();
			buf.append("This is the compressed content!  Created by John Gray\n");
			buf.append("Compression algorithm: " + algorithm.getName());
			buf.append("\n\n");
			compressedContent.setText(buf.toString());
		} else {
			compressedContent.setDataHandler(dataHandler);
		}

		try {
			compressedContent.setCompressionAlgorithm(algorithm);
		} catch (NoSuchAlgorithmException ex) {
			throw new MessagingException(
					"Compression algorithm not supported: " + ex.getMessage());
		}

		msg.setContent(compressedContent, compressedContent.getContentType());
		// let the CompressedContent update some message headers
		compressedContent.setHeaders(msg);

		return msg;
	}

	public Message addCompressionEnvelope(Message msgtowrap, Session session, AlgorithmID algorithm)
			throws MessagingException {

		String subject = "IAIK-S/MIME: Compressed [" + algorithm.getName()
				+ "]";

		Message msg = createMessage(session);
		msg.setSubject(subject);

		//These are not needed since they are being wrapped in a
		//compressed content object
		msgtowrap.removeHeader("Date");
		msgtowrap.removeHeader("From");
		msgtowrap.removeHeader("to");
		msgtowrap.removeHeader("Subject");
		msgtowrap.saveChanges();
		
		CompressedContent compressedContent = new CompressedContent();
		//CompressedContent compressedContent = createCompressedContent(session, new DataHandler(msgtowrap,msgtowrap.getContentType()),algorithm);
		// Datahandler gets set when setContent is called
		compressedContent.setContent(msgtowrap, msgtowrap.getContentType());
		//Null provider is example provider which provides additional algorithms, if not
		//set, only the default zlib_compress algorithm can be used.
		NullCompression nullprovider = new NullCompression();
		compressedContent.setCompressionProvider(nullprovider);

		// Check if the dataHandler of the compressed content was set when the Message content
		// was set.  If no datahandler is set, then default to a text-based datahandler so
		// that the Message can still be sent.
		if (compressedContent.getDataHandler() == null) {
			System.out.println("DataHandler for message passed in is Null!");
			StringBuffer buf = new StringBuffer();
			buf.append("For some reason, the message which was passed into this object could not set the Content Datahandler.  Therefore, the message could not be wrapped properly!\n");
			buf.append("Compression algorithm: " + algorithm.getName());
			buf.append("\n\n");
			compressedContent.setText(buf.toString());	
		} 

		try {
			compressedContent.setCompressionAlgorithm(algorithm);
		} catch (NoSuchAlgorithmException ex) {
			throw new MessagingException(
					"Compression algorithm not supported: " + ex.getMessage());
		}

		compressedContent.setHeaders(msg);
		msg.setContent(compressedContent, compressedContent.getContentType());
		// let the CompressedContent update some message headers
		

		return msg;
	}

	public static void main(String[] argv) throws Exception {
		(new SMimeSend()).start();
		iaik.utils.Util.waitKey();
		return;
	}

	private String[] getPKIadress() {
		String[] info = new String[2];
		try {
			String filename = properties.getString(section, "inifile");
			IniFile ini = new IniFile(filename);
			String newline = ini.getString("Entrust Settings", "Server", "");
			if (newline.equals("")) {
				System.out.println("The Server address not found.");
			} else {
				//we found the entry for Server address
				System.out.println("we found the entry for Server address");
				info[0] = newline.substring(0, newline.indexOf("+"));
				info[1] = newline.substring((newline.indexOf("+") + 1));
				System.out.println("Server: " + info[0]);
				System.out.println("Port: " + info[1]);
			}
		} catch (FileNotFoundException ex) {
			System.out.println("File Entrust.ini is missing: "
					+ ex.getMessage());
		}
		return info;
	}
}
