//===========================================================================
//
// Copyright (c)  2001-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.ssl.multithread;

import iaik.x509.X509Certificate;

import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.X509KeyManager;
import javax.net.ssl.X509TrustManager;

import com.entrust.toolkit.User;
import com.entrust.toolkit.credentials.StreamProfileReader;
import com.entrust.toolkit.security.provider.Entrust;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.util.IniFile;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.CertVerifier;
import com.entrust.toolkit.x509.jsse.JSSEUtils;
import com.entrust.toolkit.x509.jsse.JSSEX509KeyManager;
import com.entrust.toolkit.x509.jsse.JSSEX509TrustManager;

/**
 * This class implements the server part of the multithreading example. To try
 * it, run both the server and the client.
 * <p>
 * This class instantiates an SSLServerSocket and waits for incoming
 * connections. When a client connects, it sets up three threads: one to write
 * data, one to read data, and one to periodically initiate a renegotiate. The
 * synchronization performed by the library makes sure that the three threads
 * accessing the socket at the same time do not interfere with one another.
 * 
 * @see Client
 */
public class Server {

    private final static boolean DEBUG = true;

    private final static long TIME_RENEGOTIATE = (long) 1.4 * 1000;

    static IniFile properties = null;
    static String section = null;
    static String profileName = null;
    static String pwd = null;
    static String port = null;

    public static void main0(String args[]) throws Throwable {

        try {
            // get the properties from https.ini
            // properties = new IniFile("data/ssl/multithread.ini");
            properties = new IniFile("data/ssl/multithread.ini");

            // get Section
            section = new String("multithread");

            // get all the properties in order to run this Example
            profileName = properties.getString(section, "epfServer");
            pwd = properties.getString(section, "pwd");
            port = properties.getString(section, "port");

            // Initialize the security providers
            Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);

            // Create a SSL server context
            SSLContext context = JSSEUtils.getSSLContext();

            // Create an Entrust user, whose credentials we will use
            // for client authentication.
            User user = new User();

            // Create a SecureStringBuffer Object for the password
            SecureStringBuffer password = new SecureStringBuffer(new StringBuffer(pwd), 0);

            // Log into the Entrust user to access the credentials.
            System.out.println("log in");
            StreamProfileReader cr = null;
            cr = new StreamProfileReader(new FileInputStream(profileName));
            user.login(cr, password);

            X509Certificate[] certChain = new X509Certificate[2];
            certChain[0] = user.getVerificationCertificate();
            certChain[1] = user.getCaCertificate();

            // Create a KeyManager
            X509KeyManager kmgr = new JSSEX509KeyManager(certChain, user.getSigningKey());

            // Use Entrust certificate validation to only trust CAs through
            // our trust model.

            // Create a TrustManager
            X509Certificate rootCert = certChain[certChain.length - 1];
            CertVerifier certVerifier = new CertVerifier(rootCert, user.getDirectory(), user.getClientSettings());

            X509TrustManager tmgr = new JSSEX509TrustManager(certVerifier);

            // Initialize the SSLContext
            context.init(new X509KeyManager[] { kmgr }, new X509TrustManager[] { tmgr },
                    Entrust.getDefaultSecureRandomInstance());

            javax.net.ssl.SSLServerSocket serverSocket = (javax.net.ssl.SSLServerSocket) context
                    .getServerSocketFactory().createServerSocket(Integer.valueOf(port).intValue());

            String[] suites = new String[] { "SSL_DHE_RSA_WITH_DES_CBC_SHA", "SSL_DHE_RSA_WITH_3DES_EDE_CBC_SHA",
                    "TLS_DHE_RSA_WITH_AES_128_CBC_SHA", "TLS_DHE_RSA_WITH_AES_256_CBC_SHA" };

            serverSocket.setEnabledCipherSuites(suites);

            // Request for a client authentication
            serverSocket.setNeedClientAuth(true);

            System.out.println("server Waiting for connections on port " + serverSocket.getLocalPort() + "...");
            SSLSocket socket = (SSLSocket) serverSocket.accept();
            socket.setSoTimeout(20 * 1000);

            System.out.println("server Accepted connection from " + socket.getInetAddress());
            serverSocket.close();

            socket.startHandshake();
            System.out.println("server Initial handshake ok.");

            Feeder feeder = new Feeder(socket.getOutputStream(), "SFeeder");
            Reader reader = new Reader(socket.getInputStream(), "SReader");

            feeder.start();
            reader.start();

            int n = 0;

            String[] enabled = socket.getEnabledCipherSuites();
            List<String> cipherList = new ArrayList<String>();
            Collections.addAll(cipherList, enabled);

            ByteArrayOutputStream debugData = null;

            try {
                while (n < 20) {
                    feeder.checkError();
                    reader.checkError();

                    debugData = new ByteArrayOutputStream();

                    String activeCipher = socket.getSession().getCipherSuite();
                    System.out.println("activeCipher:" + activeCipher);

                    for (int i = 0; i < cipherList.size(); i++) {
                        String item = cipherList.get(i);
                        if (item.equals(activeCipher)) {
                            cipherList.remove(i);
                        }
                    }
                    socket.setEnabledCipherSuites((String[]) cipherList.toArray(new String[0]));

                    if (socket.isConnected()) {
                        System.out.println("server socket open");
                    } else {
                        System.out.println("server socket closed");
                    }

                    socket.startHandshake();
                    System.out.println("server " + n + " handshake ok.");

                    cipherList.add(activeCipher); // reordering, putting to
                                                  // bottom

                    n++;
                    System.out.print("server Renegotiates: " + n + ", Reads: " + reader.getCount() + ", Writes: "
                            + feeder.getCount() + "\r");
                    System.out.flush();

                    Thread.sleep(TIME_RENEGOTIATE);
                }
            } finally {
                feeder.setActive(false);
                reader.setActive(false);
                socket.close();
                serverSocket.close();
                System.out.println();
                if (debugData != null) {
                    System.out.println(new String(debugData.toByteArray()));
                }
            }
        } catch (Exception e) {
            System.out.println("server exception: " + e.getMessage());
            System.out.println();
            e.printStackTrace();
        }
    }

    public static void main(String args[]) {
        try {
            main0(args);
            System.out.println("server Finished.");
        } catch (Throwable e) {
            System.err.println("Unexpected server exception:");
            e.printStackTrace(System.err);
        }
    }
}
