//===========================================================================
//
// Copyright (c)  2003-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.timestamping;

import iaik.utils.Util;
import iaik.x509.X509Certificate;

import com.entrust.toolkit.User;
import com.entrust.toolkit.asn1.structures.AlgorithmID;
import com.entrust.toolkit.asn1.tsp.TimeStampToken;
import com.entrust.toolkit.credentials.FilenameProfileReader;
import com.entrust.toolkit.security.provider.Initializer;
import com.entrust.toolkit.timestamp.HttpTimeStampTransport;
import com.entrust.toolkit.timestamp.TimeStampClient;
import com.entrust.toolkit.timestamp.TimeStampTransport;
import com.entrust.toolkit.timestamp.TimeStampVerifier;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.x509.CertVerifier;

/**
 * This example shows how to request a time-stamp for any arbitrary data.
 * 
 * <P>
 * The time-stamp received from the TimeStamp Authority (TSA) is automatically 
 * verified and validated once it has been received.
 * </P>
 */
public class RequestTimeStamp {

    /******************************************************************************/
    /*                                  CONSTANTS                                 */
    /******************************************************************************/

    /** The data the time-stamp is being requested for. */
    private static final byte[] DATA_TO_BE_TIMESTAMPED = "Data to be time-stamped".getBytes();

    /******************************************************************************/
    /*                               SAMPLE PROGRAM                               */
    /******************************************************************************/

    public static void main(String[] args) {
        // Check arguments
        int size = args.length;
        if (size < 3 || size > 4) {
            System.out.println("Usage: RequestTimeStamp <epf> <pwd> <tsa's url>");
            System.out.println("\nUsage: RequestTimeStamp <epf> <pwd> <tsa's url> <tsa's base64 encoded cert>");
            System.exit(1);
        }

        // Parse the TSA certificate if it was provided
        X509Certificate tsaCert = null;
        if (size == 4) {
            Initializer.getInstance().setProviders(Initializer.MODE_NORMAL);
            try {
                tsaCert = new X509Certificate(Util.Base64Decode(Util.toASCIIBytes(args[3])));
            } catch (Exception e) {
                System.out.println("Invalid TSA certificate");
                e.printStackTrace();
            }
        }

        try {
            // Login the user (offline)
            System.out.print("Logging in user... ");
            User user = new User();
            FilenameProfileReader reader = new FilenameProfileReader(args[0]);
            SecureStringBuffer password = new SecureStringBuffer(new StringBuffer(args[1]));
            user.login(reader, password);
            System.out.println("DONE\n");

            // Add the TSA certificate as a trusted certificate
            // This must be done if the user does not share a trust relationship
            // with the TSA by some other means such as cross-certification. 
            CertVerifier certVerifier = (CertVerifier) user.getCertVerifier();
            if (tsaCert != null) {
                System.out.print("Adding TSA certificate as a trusted certificate... ");
                user.getClientSettings().setAllowCAPAB(true);
                certVerifier.getCertificateStore().addTrustedCertificate(tsaCert);
                System.out.println("DONE\n");
            }

            // Request a time stamp
            System.out.print("Requesting a time-stamp... ");
            TimeStampVerifier timeStampVerifier = new TimeStampVerifier((CertVerifier) user.getCertVerifier());
            TimeStampTransport transport = new HttpTimeStampTransport(args[2]);
            TimeStampClient timeStampClient = new TimeStampClient(transport, timeStampVerifier);
            // Optionally, the hash algorithm can be specified (default: SHA1)
            timeStampClient.setHashAlgorithm(AlgorithmID.MessageDigestAlgs.md5);
            // Optionally, the TSA certificate can be requested (or not) in the 
            // response from the time-stamp authority (default: request TSA cert)
            timeStampClient.setRequestTSACert(Boolean.TRUE);
            TimeStampToken timeStampToken = timeStampClient.requestTimeStampToken(DATA_TO_BE_TIMESTAMPED);
            System.out.println("DONE\n");

            // Display the time-stamp
            System.out.println("Time-stamp contents - ");
            System.out.println(timeStampToken.getTstInfo() + "\n");
            System.out.println("Entire time-stamp token contents - ");
            System.out.println(timeStampToken);
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }

    /******************************************************************************/
    /*                                CONSTRUCTORS                                */
    /******************************************************************************/

    /**
     * The default constructor.
     * <P></P>
     */
    private RequestTimeStamp() {
    }
}