//===========================================================================
//
// Copyright (c)  2002-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.xml.decryptionTransform;

import java.io.InputStream;
import java.util.Vector;

import iaik.ixsil.core.Verifier;
import iaik.ixsil.exceptions.SignatureException;
import iaik.ixsil.init.IXSILInit;
import iaik.ixsil.util.ExternalReferenceResolverImpl;
import iaik.ixsil.util.URI;

import com.entrust.toolkit.KeyAndCertificateSource;
import com.entrust.toolkit.User;
import com.entrust.toolkit.examples.xml.utils.Utils;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.xencrypt.init.XMLEInit;

/**
 * This class demonstrates how to verify an XML signature that contains encrypted
 * content.
 * <P>
 * As it verifies an XML signature performed over content that was subsequently
 * encrypted, the Toolkit performs the required decryptions internally.</P>
 * <P>
 * A Decryption Transform must have been inserted into the &lt;Signature&gt;
 * when it was created, so that the Toolkit can determine which content was
 * encrypted before it was signed.  (The verifier must not decrypt that content,
 * of course.)  The sample program <i>SignDecryptionTransform.java</i> shows how
 * to insert a Decryption Transform when you create an XML signature.</P>
 * <P>
 * The Toolkit's implementation of the Decryption Transform follows the specification in
 * <a href="http://www.w3.org/TR/2001/WD-xmlenc-decrypt-20011018">Decryption Transform
 * for XML Signature, W3C Working Draft 18 October 2001</a>.
 * </P>
 *@see SignDecryptionTransform
 *@see com.entrust.toolkit.xencrypt.core.TransformImplDecryption
 */
public class VerifyDecryptionTransform
{
  /**
   * An initializer for the Toolkit's XML encryption
   */
  private XMLEInit m_initializer;

  /**
   * A User who can decrypt the encrypted content
   */
  private User m_user;

  /**
   * The main method, which demonstrates signature verification.
   */
  public static void main(String[] args) throws Exception
  {
    // Instantiate this example
    VerifyDecryptionTransform verifyExample = new VerifyDecryptionTransform(args);

    // Optional -- display the XML
    Utils.displayURL(args[3]);

    // Verify an XML Signature
    if(!verifyExample.validateSignature(args[3])) {
        System.out.println("Not Verified !");
    }
  }

  /**
   * Constructor for this example. Logs into a User and initializes the
   * the Toolkit for working with XML signatures and XML encryption.
   */
  public VerifyDecryptionTransform(String[] args) throws Exception
  {
    if (args.length != 4)
    {
      System.out.println("\nPlease specify four command line parameters:");
      System.out.println("  (1) Path and filename of an Entrust user profile");
      System.out.println("  (2) Profile password");
      System.out.println("  (3) The URL of an IXSIL initialization properties file\n\t" +
                                "e.g. \"file:/C:/etjava/examples/xml/init/properties/init.properties\"\n\t" +
                                "e.g. \"http://hostname/init.properties\"");
      System.out.println("  (4) URI of the XML signature to be validated\n\t" +
                                "e.g. \"file:///C:/etjava/examples/xml/DecryptTransformTestSigned.xml\"\n\t" +
                                "e.g. \"file:DecryptTransformTestSigned.xml\"\n\t" +
                                "e.g. \"http://hostname/signature.xml\"");
      System.exit(0);
    }

    // Initialize a user
    m_user = Utils.login(args[0], new SecureStringBuffer(new StringBuffer(args[1])));

    // Initialize the Toolkit for XML signatures
    URI init = new URI(args[2]);
    System.out.println("Initializing IXSIL properties from \"" + init + "\"...");
    IXSILInit.init(init);

    // Initialize the Toolkit for XML encryption
    ExternalReferenceResolverImpl res = new ExternalReferenceResolverImpl(init);
    m_initializer = new XMLEInit(res.resolve(init));
  }

  /**
   * Verifies an XML signature that contains encrypted content.
   *
   * @param url an URL that points to a resource that contains an XML Signature <p>
   *            e.g. "file:///C:/etjava/examples/xml/signedData.xml" <p>
   *            e.g. "http://host/signedData.xml"
   */
  public boolean validateSignature(String url)
  throws Exception
  {
    URI baseURI = new URI(url);

    // Open an input stream to the XML document that contains the Signature
    System.out.println("Opening \"" + url + "\"");

    // Optional:  Get the signed document by resolving a URI.  If your
    // application already has an InputStream to the document, set the
    // base URI to null in the Verifier constructor.
    ExternalReferenceResolverImpl res = new ExternalReferenceResolverImpl(baseURI);
    InputStream istream = res.resolve(baseURI);

    // Specify an XPath that selects the Signature element.  The toolkit's
    // XML signature examples set the Signature ID to 'Signature001'.
    String signatureSelector = "//*[@Id=\"Signature001\"]";

    // Create a Verifier.
    // Note: If your application reads the XML document directly from an InputStream,
    //       so there is no base URI, set the 'baseURI' parameter to 'null'.
    System.out.println("Creating a Verifier...");
    Verifier verifier = new Verifier(istream, baseURI, signatureSelector, null);

    // Set a trust manager to validate the X509 verification certificate
    Utils.setTrustmanager(verifier, new KeyAndCertificateSource(m_user)) ;

    // Initialize the verifier for performing any required decryptions
    verifier.setXMLEinitializer(m_initializer);
    verifier.setUser(m_user);

    // Attach another User, if required

    // Verify the signature
    System.out.println("Verifying...");
    try {
        verifier.getSignature().verify();
        System.out.println("Verified !");

        // The signature has been verified, but the public key might have come merely
        // from a KeyValue in the XML Signature, not a valid certificate.
        //
        // NOTE: To disable IXSIL's use of KeyValue information, remove that subElement
        //       setting from the 'keymanager.properties' properties file.   This will
        //       ensure that the public verification key comes from a valid certificate.
        //       If you do that, you can remove the following line;
        Utils.validatePublicKey(verifier.getSignature());

        return true;  // success
    }
    catch (SignatureException e) {
        System.out.println("Signature could not be verified !");
        e.printStackTrace();
    }

    // The verifier might have been unable to decrypt some of the encrypted content,
    // because it could not access the required private keys.  List any Users that
    // the verifier needed but the application did not set.
    Vector userNames = verifier.getRequiredUsers();
    if(userNames.size() == 0) {
        System.out.println("All required Users were attached to the verifier");
        return false;  // failure
    }

    // A real application might set these additional User instances and repeat the verify operation.
    System.out.println("\nTo verify this signature, attach these Users to the Verifier: ");
    for(int i=0; i<userNames.size(); i++) {
        System.out.println("\"" + userNames.elementAt(i) + "\"");
    }
    return false;  // failure
  }
}
