//===========================================================================
//
// Copyright (c)  2002-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.xml.encrypt;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Vector;

import org.w3c.dom.Element;

import iaik.ixsil.exceptions.InitException;
import iaik.ixsil.init.IXSILInit;
import iaik.ixsil.util.URI;
import iaik.x509.X509Certificate;

import com.entrust.toolkit.KeyAndCertificateSource;
import com.entrust.toolkit.Trustmanager;
import com.entrust.toolkit.User;
import com.entrust.toolkit.examples.xml.utils.Utils;
import com.entrust.toolkit.exceptions.UserBadPasswordException;
import com.entrust.toolkit.exceptions.UserNotLoggedInException;
import com.entrust.toolkit.exceptions.UserFatalException;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.xencrypt.core.Decryptor;
import com.entrust.toolkit.xencrypt.core.EncryptedElementSet;
import com.entrust.toolkit.xencrypt.core.Encryptor;
import com.entrust.toolkit.xencrypt.exceptions.DecryptorException;
import com.entrust.toolkit.xencrypt.exceptions.EncryptionHandlerException;
import com.entrust.toolkit.xencrypt.exceptions.XMLEInitException;
import com.entrust.toolkit.xencrypt.init.XMLEInit;

/**
 * This class demonstrates how to use the Toolkit to encrypt external binary 
 * data to an EncryptedData element, and decrypt that element.
 * <p>
 * This simple example encrypts a binary data file, set the EncryptedData
 * as the top-level element in a new XML document, then decrypt that element,
 * and the returned cleartext octet sequence is written to a file.
 *
 */
public class EncryptArbitraryData {

    /**
    * The XMLE initialization object
    */
    public static XMLEInit m_initializer = null ;

    /**
    * The user that will validate encryption certificates when encrypting.
    */
    public static User m_sender = null ;

    /**
    * The recipient, who has the private decryption key
    */
    public static User m_recipient = null  ;

    /**
    * The Encryptor instance
    */
    public static Encryptor m_encryptor = null ;

    /**
    * Encryption certificate of recipient
    */
    public static X509Certificate m_recipientCertificate ;

    /**
    * Encryption certificate of recipient
    */
    public static X509Certificate m_senderCertificate ;

    /**
    * The result element of external binary data encryption
    */
    public static Element m_encryptedDataElement ;

    /**
    * main method -- Log in, encrypt external binary data, decrypt it
    */
	public static void main(String args[])
	throws FileNotFoundException,
	       IOException,
	       UserNotLoggedInException,
	       EncryptionHandlerException,
	       XMLEInitException,
	       InitException,
	       Exception
	{
	    // Get the command line arguments
        if (args.length != 7 && args.length != 8 )
        {
            System.out.println("\nPlease specify the following command line parameters:");

            System.out.println("  (1) Path and filename of an Entrust user profile representing the sender");
            System.out.println("  (2) Path and filename of an Entrust user profile representing the recipient");
            System.out.println("  (3) Profile password (same for both user profiles)");
            System.out.println("  (4) The URL of an initialization properties file\n\t" +
                                      "e.g. \"file:/C:/etjava/examples/xml/init/properties/init.properties\"\n\t" +
                                      "e.g. \"init.properties\"\n\t" +
                                      "e.g. \"http://hostname/init.properties\"");
            System.out.println("  (5) The URL of a binary file to be encrypted");
            System.out.println("  (6) Filename of the encrypted data");
            System.out.println("  (7) Filename of the decrypted data");
            System.out.println("  (8) [Optional] The URL to store the encrypted cipher octet externally\n\t" +
                                      "e.g. \"file:/C:/etjava/examples/xml/encrypted/ciphertext.bin\"\n\t" +
                                      "e.g. \"file:/C:../encrypted/ciphertext.bin\"\n\t" +
                                      "e.g. \"http://hostname/ciphertext.bin\"");
            System.exit(0);
        }

	    int argsNumber = args.length ;

        int index = 0 ;
		String sender         = args[index++];
		String recipient      = args[index++];
		String password       = args[index++];
		String propertiesURL  = args[index++];
		String binaryDataURL  = args[index++];
		String encryptedFile  = args[index++] ;
		String decryptedFile  = args[index++];
		String ciphertextURL  = null;

		if(args.length == 8) {
		    ciphertextURL = args[index++];
		}

	    // Get the properties for the XMLE system.
        URI initProps = new URI(propertiesURL);

	    // Initialize IXSIL because XMLE invokes a few of its utility methods.
        System.out.println("Initializing IXSIL properties from \"" + initProps + "\"...");
        IXSILInit.init(initProps);

	    // Initialize XMLE.
        System.out.println("Initializing XMLE properties from \"" + initProps + "\"...");
        m_initializer = new XMLEInit(initProps);

		// Log in to the Toolkit
        m_sender = Utils.login(sender, new SecureStringBuffer(new StringBuffer(password)));

        // OPTIONAL:  required if the sender will encrypt for itself
        m_senderCertificate = m_sender.getEncryptionCertificate();

		// Log in to the Toolkit
        m_recipient = Utils.login(recipient, new SecureStringBuffer(new StringBuffer(password)));

        // Get the encryption certificate of a recipient
        m_recipientCertificate = m_recipient.getEncryptionCertificate();

        // Encrypt an external binary file
        encrypt(binaryDataURL, encryptedFile, ciphertextURL) ;

        // Decrypt and write to a binary file
        decrypt(encryptedFile, decryptedFile) ;

        // Done.
        m_sender.logout();
        m_recipient.logout();
	}

    /**
    * Creates a DOM document.  Encrypts the document and writes the result to a file.
    * @param encryptedFile The file name of the encrypted XML document
    */
    private static void encrypt(String binaryDataURL, String encryptedFile, String ciphertextURL)
 	throws EncryptionHandlerException,
            FileNotFoundException,
            UserNotLoggedInException,
            UserFatalException,
            IOException
    {
	    // Create a new XML document to contain encrypted data
	    m_encryptor = new Encryptor(m_initializer);

        // Open an input stream to the binary data, a file in this example
        System.out.println("Encrypting \"" + binaryDataURL + "\" for \"" +
                            m_recipientCertificate.getSubjectDN().getName() + "\"");

        // Create a set into which is added all the DOMs elements to be encrypted
        EncryptedElementSet set = new EncryptedElementSet(m_encryptor);

        // Just one element in this simple example
        set.addElement(binaryDataURL);

        // Store ciphertext externally
        if(ciphertextURL != null){
            System.out.println("Store ciphertext externally at \"" + ciphertextURL + "\"");
            m_encryptor.setCipherURI(binaryDataURL, ciphertextURL);
        }

	    // Provide a trust manager, so recipients' certificates are validated
        m_encryptor.setTrustmanager(new Trustmanager(new KeyAndCertificateSource(m_sender)));;

        // Encrypt all elements in the set (actually, just one) for this recipient.
        set.addRecipient(m_recipientCertificate);

        // OPTIONAL: Sender might add itself as a recipient (not required)
        // set.addRecipient(m_senderCertificate);

        // Encrypt
        m_encryptor.encrypt();

        // retrieve the EncryptedData element
        Element encryptedDataElement = m_encryptor.getEncryptedDataElement(binaryDataURL);

        if(ciphertextURL != null){
            // In this example we write the ciphertext into a file named "ciphertext.bin" in the local folder,
            // so the ciphertext uri has to point to this location
            // The application should have their code to store the ciphertext bytes.
            FileOutputStream fos = new FileOutputStream(new File("ciphertext.bin"));
            fos.write(m_encryptor.getCipherText(ciphertextURL));
            fos.close();
        }

        // Write the encrypted XML document
        System.out.println("Writing to: " + encryptedFile);
        m_encryptor.toOutputStream(new FileOutputStream(encryptedFile));

    }

    /**
    * Decrypts the document and writes the result to a file.
    *
    * @param encryptedFile The file name of an encrypted XML document
    * @param decryptedFile The file name of the decrypted XML document
    */
    private static void decrypt(String encryptedFile, String decryptedFile)
    throws EncryptionHandlerException,
           FileNotFoundException,
           UserNotLoggedInException,
           UserBadPasswordException,
           IOException
    {

        Decryptor decryptor = null ;

	    System.out.println("Decrypting...");

        // Create a Decryptor instance
        decryptor = new Decryptor(m_initializer, new FileInputStream(encryptedFile));

        // Display recipient names
        showRecipients(decryptor);

        // Attach a user who can decrypt, based on the recipient identities
	    decryptor.addUser(m_recipient);

        decryptor.decrypt();

        // get the decrypted octet sequence
        byte[] decryptedByte = (byte[]) decryptor.getDecryptedBinary().elementAt(0);

        // Write the decrypted document
        System.out.println("Writing to: " + decryptedFile);
        FileOutputStream fos = new FileOutputStream(decryptedFile);
        fos.write(decryptedByte);
        fos.close();
    }


    /**
    * Displays recipient names.  This basic example has just one recipient.  In general
    * this method allows the application to identify which User instances have the
    * private decryption keys that will decrypt a particular document.
    *
    * @param decryptor The Decryptor instance
    */
    private static void showRecipients(Decryptor decryptor)
    throws DecryptorException,
           UserNotLoggedInException,
           UserBadPasswordException
    {
        Vector recipients = decryptor.getRecipients();
        for(int i=0; i<recipients.size(); i++) {
            System.out.println("\"" + recipients.elementAt(i) + "\"" + " is a recipient.");
        }

        // The user who will decrypt must be one these recipients, of course.
        System.out.println("\"" + m_recipient.getVerificationCertificate().getSubjectDN().getName() +
                           "\"" + " will decrypt.");
    }

}