//===========================================================================
//
// Copyright (c)  2002-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.xml.encrypt;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.security.cert.CertificateException;
import java.util.Vector;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import iaik.ixsil.exceptions.InitException;
import iaik.ixsil.init.IXSILInit;
import iaik.ixsil.util.URI;
import iaik.x509.X509Certificate;

import com.entrust.toolkit.KeyAndCertificateSource;
import com.entrust.toolkit.Trustmanager;
import com.entrust.toolkit.User;
import com.entrust.toolkit.examples.xml.utils.Utils;
import com.entrust.toolkit.exceptions.UserBadPasswordException;
import com.entrust.toolkit.exceptions.UserFatalException;
import com.entrust.toolkit.exceptions.UserNotLoggedInException;
import com.entrust.toolkit.util.SecureStringBuffer;
import com.entrust.toolkit.xencrypt.core.Decryptor;
import com.entrust.toolkit.xencrypt.core.EncryptedElementSet;
import com.entrust.toolkit.xencrypt.core.Encryptor;
import com.entrust.toolkit.xencrypt.exceptions.DecryptorException;
import com.entrust.toolkit.xencrypt.exceptions.EncryptedElementSetException;
import com.entrust.toolkit.xencrypt.exceptions.EncryptionHandlerException;
import com.entrust.toolkit.xencrypt.exceptions.EncryptorException;
import com.entrust.toolkit.xencrypt.exceptions.XMLEInitException;
import com.entrust.toolkit.xencrypt.init.XMLEConstants;
import com.entrust.toolkit.xencrypt.init.XMLEInit;

/**
 * This class demonstrates how to use the Toolkit to encrypt and decrypt XML 
 * documents.  This example demonstrates how to control EncryptedKeys and how 
 * to encrypt for multiple recipients.
 * <p>
 * This example demonstrates the use of the EncryptedElementSet class, a convenience
 * which allows your application easily to specify groups of DOM elements that are
 * to be encrypted for particular groups of recipients.  In addition, all DOM elements
 * in an EncryptedElementSet will be encrypted by the same symmetric key, so relatively
 * few private key operations are required when decrypting (fewer EncryptedKey elements
 * are needed);  therefore, your application might use an EncryptedElementSet
 * even if the document has only one recipient, but that recipient will encrypt
 * several elements within it.
 */
public class EncryptExtended {

    /**
    * The XMLE initialization object
    */
    public static XMLEInit m_initializer = null ;

    /**
    * The user that will validate encryption certificates when encrypting.
    */
    public static User m_sender = null ;

    /**
    * A User instance for Recipient 'A', which contains the private decryption key
    */
    public static User m_recipientA = null  ;

    /**
    * A User instance for Recipient 'B', which contains the private decryption key
    */
    public static User m_recipientB = null  ;

    /**
    * The Encryptor instance
    */
    public static Encryptor m_encryptor = null ;

    /**
    * Encryption certificate for recipient A
    */
    public static X509Certificate m_certA ;

    /**
    * Encryption certificate for recipient B
    */
    public static X509Certificate m_certB ;

    /**
    * Each set contains DOM elements to be encrypted for a particular group
    * of recipients: Recipient A alone, Recipient B alone, or both recipients.
    */
    public static EncryptedElementSet[] m_sets = null  ;

    /**
     * XML elements with this tag name will be encrypted for both 'A' and 'B'.
     */
    private static final String ENCRYPT = "Encrypt";

    /**
     * The content of XML elements with this tag name will be encrypted for both 'A' and 'B'.
     */
    private static final String ENCRYPT_CONTENT = "EncryptContent";

    /**
     * XML elements with this tag name will be encrypted for 'A'.
     */
    private static final String ENCRYPT_FOR_RECIPIENT_A = "EncryptForA";

    /**
     * The content of XML elements with this tag name will be encrypted for 'A'.
     */
    private static final String ENCRYPT_CONTENT_FOR_RECIPIENT_A = "EncryptContentForA";

    /**
     * XML elements with this tag name will be encrypted for 'B'.
     */
    private static final String ENCRYPT_FOR_RECIPIENT_B = "EncryptForB";

    /**
     * The content of XML elements with this tag name will be encrypted for 'B'.
     */
    private static final String ENCRYPT_CONTENT_FOR_RECIPIENT_B = "EncryptContentForB";

    /**
    * main method -- Log in, encrypt a document, decrypt the encrypted document.
    */
	public static void main(String args[])
	throws FileNotFoundException,
	       IOException,
	       CertificateException,
	       UserNotLoggedInException,
	       EncryptionHandlerException,
	       XMLEInitException,
	       InitException,
	       Exception
	{
	    // Get the command line arguments
        if (args.length < 7 || args.length > 9)
        {
            System.out.println("\nPlease specify the following command line parameters:");

            System.out.println("  (1) Path and filename of an Entrust user profile for the user that will encrypt");
            System.out.println("  (2) Path and filename of an Entrust user profile for recipient 'A'");
            System.out.println("  (3) Path and filename of an Entrust user profile for recipient 'B'");
            System.out.println("  (4) Profile password (for simplicity, it's the same for all three user profiles)");
            System.out.println("  (5) The URL of an initialization properties file\n\t" +
                                      "e.g. \"file:/C:/etjava/examples/xml/init/properties/init.properties\"\n\t" +
                                      "e.g. \"init.properties\"\n\t" +
                                      "e.g. \"http://hostname/init.properties\"");
            System.out.println("  (6) A file name for the encrypted XML document");
            System.out.println("  (7) A file name for the decrypted XML document");
            System.out.println("  (8) [Optional - an encryption certificate for recipient 'A']\n      " +
                                      "If none is provided (enter \"\"), the sample will take it from that recipient's user profile");
            System.out.println("  (9) [Optional - an encryption certificate for recipient 'B']\n      " +
                                      "If none is provided (enter \"\"), the sample will take it from that recipient's user profile");
            System.exit(0);
        }

	    int argsNumber = args.length ;

        int index = 0 ;
		String sender         = args[index++];
		String recipientA     = args[index++];
		String recipientB     = args[index++];
		String password       = args[index++];
		String propertiesURL  = args[index++];
		String encryptedFile  = args[index++] ;
		String decryptedFile  = args[index++];

	    // Get the properties for the XMLE system.
        URI initProps = new URI(propertiesURL);

	    // Initialize IXSIL because XMLE invokes a few of its utility methods.
        System.out.println("Initializing IXSIL properties from \"" + initProps + "\"...");
        IXSILInit.init(initProps);

	    // Initialize XMLE.
        System.out.println("Initializing XMLE properties from \"" + initProps + "\"...");
        m_initializer = new XMLEInit(initProps);

		// Log in the user that will encrypt
        m_sender = Utils.login(sender, new SecureStringBuffer(new StringBuffer(password)));

		// Log in the recipients here, in case this sample must take the recipients'
		// encryption certificates from their user profiles.  A real application will
		// obtain them from a certificate repository (see JNDIDirectory) or another source.
        m_recipientA = Utils.login(recipientA, new SecureStringBuffer(new StringBuffer(password)));
        m_recipientB = Utils.login(recipientB, new SecureStringBuffer(new StringBuffer(password)));

        // Get the encryption certificates of the recipients
		if(args.length == 9) {
            m_certA = new X509Certificate(new FileInputStream(args[index++]));
            m_certB = new X509Certificate(new FileInputStream(args[index++]));
		}
		else {
            m_certA = m_recipientA.getEncryptionCertificate();
            m_certB = m_recipientB.getEncryptionCertificate();
		}

        // This XML sample contains elements named 'EncryptForA', EncryptForB', etc.
		String documentToEncrypt  = "data/xml/encrypt/encryptForAandB.xml";

		
		
        // Encrypt
        encrypt(documentToEncrypt) ;

        // OPTIONAL: Reposition the EncryptedKey DOM elements to any place in the document,
        //           in this example, immediately preceding the first EncryptedData element.
        // moveEncryptedKeys();

        // Write the encrypted document
        System.out.println("\nWriting to: " + encryptedFile);
        m_encryptor.toOutputStream(new FileOutputStream(encryptedFile));

        // Decrypt
        decrypt(encryptedFile, decryptedFile) ;

        // Done.
        m_sender.logout();
        m_recipientA.logout();
        m_recipientB.logout();
	}

    /**
    * Encrypts an XML document
    *
    * @param xmlFile The file name of an XML document to encrypt
    */
    public static void encrypt(String xmlFile)
    throws EncryptionHandlerException, FileNotFoundException, UserNotLoggedInException,
    UserFatalException
    {

        System.out.println("\n\nEncrypting for: " + m_certA.getSubjectDN().getName());
        System.out.println("Encrypting for: " + m_certB.getSubjectDN().getName());

	    // Create an Encryptor instance
	    m_encryptor = new Encryptor(m_initializer, new FileInputStream(xmlFile));

	    // Provide a trust manager, so encryption certificates are validated
        m_encryptor.setTrustmanager(new Trustmanager(new KeyAndCertificateSource(m_sender)));

	    // Set the symmetric encryption algorithm to be AES or omit this line
	    // to let the toolkit use its default.  Additional algorithm identifiers
	    // are found in the Javadoc for class XMLEConstants.
        m_encryptor.setSymmetricAlgorithm(XMLEConstants.ALGORITHM_AES_256);

	    // Specify a value for the Id attribute of EncryptedKey elements.  Omit
	    // this line to let the toolkit set those attributes to default values.
	    // (see the Javadoc for the Encryptor class).
        m_encryptor.setEncryptedKeyBaseID("Key");

	    // Specify a value for the Id attribute of EncryptedData elements.  Omit
	    // this line to let the toolkit set those attributes to default values.
	    // (see the Javadoc for the Encryptor class).
        m_encryptor.setEncryptedDataBaseID("Data");

        // Collect DOM elements into sets to be encrypted for particular recipients
        configureRecipients();

        // Encrypt
        m_encryptor.encrypt();

        // OPTIONAL.  Display info regarding the EncryptedKey and EncryptedData elements
        // that the toolkit created.   Your application can use the DOM API to move those
        // EncryptedKey elements to different positions in the document, if desired.
        displayEncryptionInfo();
    }

    /**
     * Creates three sets of DOM elements to be encrypted respectively for Recipient A,
     * Recipient B, and for both recipients. Collects XML elements into the appropriate set;
     * in this simple example, any DOM element that has the tag "EncryptForA" is added to the
     * set for Recipient A, and so on.  Your application will have its own way of collecting
     * the DOM elements you want to encrypt for each group of recipients.
     *
     */
    public static void configureRecipients()
    throws EncryptedElementSetException, EncryptorException
    {
        // Create three sets of DOM elements into which we will add elements
        // to be encrypted for Recipient A, Recipient B, or both recipients.
        // As a demo, we'll use a different encryption algorithm for each set.
        m_sets = new EncryptedElementSet[3];

        m_encryptor.setSymmetricAlgorithm(XMLEConstants.ALGORITHM_AES_128);
        m_sets[0] = new EncryptedElementSet(m_encryptor);

        m_encryptor.setSymmetricAlgorithm(XMLEConstants.ALGORITHM_3DES);
        m_sets[1] = new EncryptedElementSet(m_encryptor);

        m_encryptor.setSymmetricAlgorithm(XMLEConstants.ALGORITHM_AES_256);
        m_sets[2] = new EncryptedElementSet(m_encryptor);

        // Elements in Set '0' will be encrypted for both recipients.
        m_sets[0].addRecipient(m_certA);
        m_sets[0].addRecipient(m_certB);

        // Only recipient 'A' can decrypt the elements we put in Set '1'.
        // Set the encryption method to PKCS#1 v1.5 for that set.
        m_encryptor.setEncryptedKeyAlgorithm(XMLEConstants.ALGORITHM_RSA);
        m_sets[1].addRecipient(m_certA);

        // Only recipient 'B' can decrypt the elements we put in Set '2'.
        // Set the encryption method to PKCS#1 v2.0 (OAEP) for that set.
        m_encryptor.setEncryptedKeyAlgorithm(XMLEConstants.ALGORITHM_RSA_OAEP);
        m_sets[2].addRecipient(m_certB);

        // Add all the DOM elements whose tag and content will be encrypted.
        getElements(ENCRYPT,                 m_sets[0], false);
        getElements(ENCRYPT_FOR_RECIPIENT_A, m_sets[1], false);
        getElements(ENCRYPT_FOR_RECIPIENT_B, m_sets[2], false);

        // Add all the DOM elements for which only the element content will be encrypted, not the tag.
        getElements(ENCRYPT_CONTENT,                 m_sets[0], true);
        getElements(ENCRYPT_CONTENT_FOR_RECIPIENT_A, m_sets[1], true);
        getElements(ENCRYPT_CONTENT_FOR_RECIPIENT_B, m_sets[2], true);
    }

    /**
     * This simple example adds all the XML elements with a given name into an
     * element set that will be encrypted for a particular group of recipients,
     * and it sets an encryption Type attribute for those elements.  Your application
     * will have its own way of specifying which elements should be encrypted for
     * each group of recipients.
     *
     * @param elementTagName The name of the elements to encrypt
     * @param set The EncryptedElementSet to which the elements should be added
     * @param contentOnly If 'true', only the element content will be encrypted, not the element tag
     */
    private static void getElements(String elementTagName, EncryptedElementSet set, boolean contentOnly)
    throws EncryptorException,
           EncryptedElementSetException
    {
        // Locate the DOM elements we would like to encrypt.

        NodeList elements = m_encryptor.getDocument().getElementsByTagName(elementTagName);
        for (int i = 0; i < elements.getLength(); i++) {

            Element e = (Element) elements.item(i);

            set.addElement(e);
            m_encryptor.setContentOnly(e, contentOnly);

            // OPTIONAL: Set the Id attributes to particular values.  Your application might do
            // this so the recipient can decrypt certain EncryptedData elements while leaving
            // others encrypted, selecting those elements by Id.
            String id = new String((new StringBuffer(elementTagName)).insert(7, "ed")) + "_" + i;
            m_encryptor.setEncryptedDataId(e, id);
        }
    }

    /**
    * As an example, moves all EncryptedKey elements ahead of the first EncryptedData element.
    */
    public static void moveEncryptedKeys()
    throws EncryptionHandlerException, FileNotFoundException
    {

	    System.out.println("\n\nMoving EncryptedKeys...");

	    // Move all the EncryptedKey elements ahead of the first EncryptedData
        Node root = m_encryptor.getDocument().getDocumentElement();

	    // Find the first EncryptedData element
        Node data = m_encryptor.getDocument().getElementsByTagNameNS(
                            XMLEConstants.NAMESPACE_URI_XMLENCRYPTION_ELEM,
                            XMLEConstants.ENCRYPTEDDATA_TAG).item(0);

	    // Find all EncryptedKey elements
        NodeList keys = m_encryptor.getDocument().getElementsByTagNameNS(
                            XMLEConstants.NAMESPACE_URI_XMLENCRYPTION_ELEM,
                            XMLEConstants.ENCRYPTEDKEY_TAG);

        for (int i = 0; i < keys.getLength(); i++) {

            Node key = keys.item(i);
            key.getParentNode().removeChild(key);
            root.insertBefore(key, data);
            System.out.println("Repositioned EncryptedKey element with Id " + ((Element) key).getAttribute("Id"));
        }
    }


    /**
     * Displays information regarding the EncryptedData and EncryptedKey DOM elements
     * the toolkit created during the encryption process.
     */
    public static void displayEncryptionInfo()
    throws EncryptedElementSetException, EncryptorException
    {
        // Get the EncryptedKeys that were created by the encrypt() operation.  This example,
        // just prints out some info regarding the elements.  Your application might use
        // this information to relocate particular EncryptedKey elements to different locations
        // in the document.  By default, XMLE appends them to the end of the document.

        for(int i=0;  i < m_sets.length; i++) {

            System.out.println();

            Element[] keys     = m_sets[i].getEncryptedKeys();
            Element[] elements = m_sets[i].getEncryptedDatas();

            System.out.println("Recipient set " + i + " contains " + elements.length +
                        " EncryptedData elements that are encrypted by " + keys.length + " EncryptedKeys" );

            for (int j=0; j<elements.length; j++) {
                System.out.println("Set:" + i + ", Element:" + j + " has Id = " + elements[j].getAttribute("Id"));
            }

            for (int j=0; j<keys.length; j++) {
                System.out.println("Set:" + i + ", Key:" + j + " has Id = " + keys[j].getAttribute("Id"));
            }
        }
    }


    /**
    * Decrypts the document and writes the result to a file.
    *
    * @param encryptedFile The file name of an encrypted XML document
    * @param decryptedFile The file name of the decrypted XML document
    */
    public static void decrypt(String encryptedFile, String decryptedFile)
    throws EncryptionHandlerException,
           FileNotFoundException,
           UserNotLoggedInException,
           UserBadPasswordException
    {

        Decryptor decryptor = null ;

	    System.out.println("\n\nDecrypting...");

        // Create a Decryptor instance
        decryptor = new Decryptor(m_initializer, new FileInputStream(encryptedFile));

        // OPTIONAL: Display recipient names
        displayRecipients(decryptor);

        // Attach a user who can decrypt, based on the recipient identities displayed above.
        //System.out.println("\"" + m_recipientA.getVerificationCertificate().getSubjectDN().getName() + "\"" + " will decrypt.");
	    decryptor.addUser(m_recipientA);

        // OPTIONAL: Attach another user.  Your application might not want to decrypt the elements
        // encrypted for certain recipients, e.g the application might be a "proxy" of some kind.
        //System.out.println("\"" + m_recipientB.getVerificationCertificate().getSubjectDN().getName() + "\"" + " will decrypt.");
	    decryptor.addUser(m_recipientB);

        // Decrypt any element that has that these Users as a recipient.
        decryptor.decrypt();

        // Write the decrypted document
        System.out.println("\nWriting to: " + decryptedFile);
        decryptor.toOutputStream(new FileOutputStream(decryptedFile));
    }

    /**
    * Displays recipient names.  This method allows the application to identify which
    * User instances have the private decryption keys.
    *
    * @param decryptor The Decryptor instance
    */
    public static void displayRecipients(Decryptor decryptor)
    throws DecryptorException,
           UserNotLoggedInException,
           UserBadPasswordException
    {
        Vector recipients = decryptor.getRecipients();
        for(int i=0; i<recipients.size(); i++) {
            System.out.println("\"" + recipients.elementAt(i) + "\"" + " is a recipient.");
        }

    }
}