//===========================================================================
//
// Copyright (c)  2000-2010 Entrust.  All rights reserved.
// 
//===========================================================================
package com.entrust.toolkit.examples.xml.soap;

import java.io.BufferedWriter;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.security.PrivateKey;

import iaik.ixsil.exceptions.IXSILException;
import iaik.ixsil.init.IXSILConstants;
import iaik.ixsil.init.IXSILInit;
import iaik.ixsil.util.URI;
import iaik.utils.RFC2253NameParserException;
import iaik.x509.X509Certificate;

import com.entrust.toolkit.KeyAndCertificateSource;
import com.entrust.toolkit.User;
import com.entrust.toolkit.examples.xml.utils.Utils;
import com.entrust.toolkit.util.SecureStringBuffer;


/**
 * SoapSigner is an example of how to use the Toolkit to sign a SOAP message in accordance 
 * with WS-Security.  Specifically, this example creates a SOAP Header element and a WS-Security
 * Security element to contain the dsig Signature element.  The signature is created in the usual way, 
 * but the verification certificate is set into the KeyInfo as a SecurityTokenReference, instead of
 * an X509Certificate element.
 * <p>
 * This example code also has methods that show how to sign a fragment of any XML document, as in 
 * <code>SoapSigner.xmlFragmentSign(...)</code>, where the fragment can be selected by an ID attribute
 * of the enclosing element or by an XPath expression.
 * <p>
 * With either method, you can select the document to sign by providing a URL or an InputStream.
 * <p>
 * Note: The WS-Security XML documents created by this example can by verified by the toolkit's
 *       Verify.java example, see <code>com.entrust.toolkit.examples.xml.sign.Verify</code>.
 *       Nothing specific to SOAP messages is needed in the verification.
 */
public class SoapSigner
{
    //~ Instance fields ////////////////////////////////////////////////////////////////////////////////////////////////////////////

    private SigningWorker m_signer = null;

    //~ Constructors ///////////////////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * Creates the SOAP signer example.
     *
     * @param args command line arguments
     *
     * @throws Exception if the initialization of the sample fails for any reason
     */
    public SoapSigner(String args[]) throws Exception
    {
        if ((args.length != 5) && (args.length != 6)) {
            System.out.println("\nPlease specify the following command line parameters, number (6) being optional:");
            System.out.println("  (1) Path and filename of an Entrust user profile");
            System.out.println("  (2) Profile password");
            System.out.println("  (3) The URL of an IXSIL initialization properties file\n\t" +
                "e.g. \"file:/C:/etjava/examples/xml/init/properties/init.properties\"\n\t" + 
                "e.g. \"init.properties\"\n\t" +
                "e.g. \"http://hostname/init.properties\"");
            System.out.println("  (4) URI of an SOAP envelope to be signed, e.g. \"file:data/xml/soap/soapSignMe.xml\"");
            System.out.println("  (5) Path and filename for the resulting XML signature file, e.g. \"data\\xml\\soap\\soapSigned.xml\"");
            System.out.println("  (6) [Id of an XML element in SOAP document (4) that identifies an element whose content will be signed, e.g. signmeId]");
            System.exit(0);
        }

        // Need the signing key and certificate to be in a KeyAndCertificateSource wrapper.
        KeyAndCertificateSource source = new KeyAndCertificateSource();

        // If you using a keystore such as P12, PFX, Entrust EPF, or java.security.KeyStore,
        // then log in here, to get access to its signing key and certificate.
        User user = Utils.login(args[0], new SecureStringBuffer(new StringBuffer(args[1])));
        PrivateKey pk = user.getSigningKey();
        X509Certificate cert = user.getVerificationCertificate();

        // In any case, provide a signing key and X.509 verification certificate from whatever source you use.
        source.setSigningInfo(pk, cert);

        // The properties file is specified by a URI.  You can provide the name of the
        // file, e.g 'init.properties', to the URI constructor.  In that case, the folder
        // that contains init.properties must be in the classpath.
        // Please refer to the comments in the 'init.properties' sample.
        URI initProps = new URI(args[2]);

        // Initialize IXSIL
        System.out.println("Initializing IXSIL properties from \"" + initProps + "\"...");
        IXSILInit.init(initProps);
        System.out.println("IXSIL is initialized.");

        m_signer = new SigningWorker(source);
    }

    //~ Methods ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * main method
     *
     * @param args command line arguments
     *
     * @throws Exception if the method fails to create a signed document
     */
    public static void main(String args[]) throws Exception
    {
        String signed = null;
        
        // Create this example
        SoapSigner soapSignExample = new SoapSigner(args);

        // Document to sign
        String soapEnvelopeUri = args[3];

        // Signed document is written to this disk file.
        String signedDocFileName = args[4];

        // Optional argument giving an ID in the SOAP document.  The content
        // at this ID will be signed.  If it is not provided, the whole document
        // will be signed in an enveloped signature.
        String toBeSignedSoapBodyById = (args.length > 5) ? args[5] : null;

        // Schema locations can be configured in IXSIL's init.properties file.
        // The example folders contain local copies of the schemas for SOAP, WS-Security, and WS-Utility.
        // Create a schemaLocations string in the format required by the parser.
        String schemaLocations = Utils.getWSschemaLocations();

        // In general, you might need a noNamespaceSchemaLocation.  In the the example SOAP message provided
        // with the toolkit, all elements have namespace prefixes, so this can be null.
        // Set a URI for the schema, e.g. file:/C:/path/mydocumentschema.xsd
        String noNamespaceSchemaLocation = null;

        if (toBeSignedSoapBodyById != null) {
            
            // Use WS-Security.   
            // Create a signature over a portion of the SOAP message (e.g. the SOAP Body).
            // and put that <Signature> in a <wsse:Security> element in the SOAP Header.
            //
            signed = soapSignExample.soapSign(toBeSignedSoapBodyById, soapEnvelopeUri, noNamespaceSchemaLocation, schemaLocations);
            System.out.println("Created a signature over content at Id \"" + toBeSignedSoapBodyById +
                "\" in the SOAP document at \"" + soapEnvelopeUri + "\"");
        } else {
            // Alternate procedure, not WS-Security.  This might be adequate for some purposes.
            // Create a signature over a portion of the XML document, and put the Signature in the SOAP body. 
            //
            // Note: If 'toBeSignedSoapBodyById' is null, the whole document is signed.  This will be faster
            // than signing selected content, because you might not need a validating parse.
            //
            String xpathSignaturePosition = "(//soap:Body[1])";
            signed = soapSignExample.xmlFragmentSign(toBeSignedSoapBodyById, soapEnvelopeUri, xpathSignaturePosition,
                                                     noNamespaceSchemaLocation, schemaLocations);
            System.out.println("An enveloped signature was placed at " + xpathSignaturePosition);
        }

        // Optional -- Display signed document.
        System.out.println("\n" + signed);
        
        // Write signed document to disk file.  
        // You can verify either kind of signature with the Verify example. 
        System.out.println("\nWriting to file: " + signedDocFileName);
        
        FileOutputStream os = new FileOutputStream(signedDocFileName);
        BufferedWriter bw = new BufferedWriter(new OutputStreamWriter(os, IXSILConstants.JAVA_ENCODING_UTF8_));
        bw.write(signed);
        bw.flush();
        bw.close();

        System.out.println("Done");
    }


    /**
     * Returns string representation of XML signature over SOAP Body. Use this method for signing the body of a SOAP envelope, as
     * explained in the WS-Security specification (http://schemas.xmlsoap.org/ws/2002/12/secext).  You must provide a schema for
     * the document. If the SOAP Header element is present, it is expected that the WS-Security Security element is also present
     * and it is a child of the SOAP Header.
     *
     * @param toBeSignedSoapBodyById Identifies SOAP Body.  Holds attribute of type "ID", as defined in
     *        http://www.w3.org/2001/XMLSchema.  Must not be null.
     * @param soapEnvelopeUri Absolute URI of XML resource containing SOAP envelope.
     * @param noNamespaceSchemaLocation Use this in cases where the schema for your document does not have a namespace (eg.
     *        "http://www.dummyUri.com/soapSchema.xsd"). May be null.
     * @param schemaLocations Space-separated list of namespace/schema pairings (eg. "http://www.dummyUri.com/myDocumentSchema
     *        http://www.dummyUri.com/soapSchema.xsd"). May be null.
     *
     * @return a String that is the signed document
     *
     * @throws IXSILException if IXSIL throws an exception when creating the signature
     * @throws IOException if there is an exception when serializing the signature to a string
     */
    public String soapSign(String toBeSignedSoapBodyById, String soapEnvelopeUri, String noNamespaceSchemaLocation,
        String schemaLocations) throws IXSILException, IOException
    {
        return m_signer.soapSign(toBeSignedSoapBodyById, soapEnvelopeUri, noNamespaceSchemaLocation, schemaLocations);
    }

    /**
     * Returns string representation of XML signature over SOAP Body. Use this method for signing the body of a SOAP envelope, as
     * explained in the WS-Security specification (http://schemas.xmlsoap.org/ws/2002/12/secext).  You must provide a schema for
     * the document. If the SOAP Header element is present, it is expected that the WS-Security Security element is also present
     * and it is a child of the SOAP Header.
     *
     * @param toBeSignedSoapBodyById Identifies SOAP Body.  Holds attribute of type "ID", as defined in
     *        http://www.w3.org/2001/XMLSchema.  Must not be null.
     * @param soapEnvelopeData XML data containing SOAP envelope.
     * @param noNamespaceSchemaLocation Use this in cases where the schema for your document does not have a namespace (eg.
     *        "http://www.dummyUri.com/soapSchema.xsd"). May be null.
     * @param schemaLocations Space-separated list of namespace/schema pairings (eg. "http://www.dummyUri.com/myDocumentSchema
     *        http://www.dummyUri.com/soapSchema.xsd"). May be null.
     *
     * @return a String that is the signed document
     *
     * @throws IXSILException if IXSIL throws an exception when creating the signature
     * @throws IOException if there is an exception when serializing the signature to a string
     */
    public String soapSignData(String toBeSignedSoapBodyById, InputStream soapEnvelopeData, String noNamespaceSchemaLocation,
        String schemaLocations) throws IXSILException, IOException
    {
        return m_signer.soapSignData(toBeSignedSoapBodyById, soapEnvelopeData, noNamespaceSchemaLocation, schemaLocations);
    }
    


    /**
     * Returns string representation of XML signature. Use this method for signing a subset of an XML document.  You must provide a
     * schema for the document.
     *
     * @param toBeSignedXmlById Identifies element for signing.  Holds attribute of type "ID", as defined in
     *        http://www.w3.org/2001/XMLSchema.  If null, this method will return an enveloped signature over the entire document.
     * @param xmlResourceUri Absolute URI of XML resource containing element for signing.
     * @param xpathSignaturePosition XPath insertion point for signature.  If null, inserts the signature as the first child of
     *        root element.
     * @param noNamespaceSchemaLocation Use this in cases where the schema for your document does not have a namespace (eg.
     *        "http://www.dummyUri.com/myDocumentSchema.xsd"). May be null.
     * @param schemaLocations Space-separated list of namespace/schema pairings (eg. "http://www.dummyUri.com/myDocumentSchema
     *        http://www.dummyUri.com/myDocumentSchema.xsd"). May be null.
     *
     * @return a String that is the signed document
     *
     * @throws IXSILException if IXSIL throws an exception when creating the signature
     * @throws IOException if there is an exception when serializing the signature to a string
     */
    public String xmlFragmentSign(String toBeSignedXmlById, String xmlResourceUri, String xpathSignaturePosition,
        String noNamespaceSchemaLocation, String schemaLocations)
    throws IXSILException, IOException, RFC2253NameParserException
    {
        return m_signer.xmlFragmentSign(toBeSignedXmlById, xmlResourceUri, xpathSignaturePosition, noNamespaceSchemaLocation,
            schemaLocations);
    }

    /**
     * Returns string representation of XML signature. Use this method for signing a subset of XML data.  You must provide a schema
     * for the data.
     *
     * @param toBeSignedXmlDataById Identifies element for signing.  Holds attribute of type "ID", as defined in
     *        http://www.w3.org/2001/XMLSchema.  If null, this method will return an enveloped signature over the entire XML
     *        resource.
     * @param xmlResourceData an input stream to XML data containing an element whose content will be signed.
     * @param xpathSignaturePosition XPath insertion point for signature.  If null, inserts the signature as the first child of
     *        root element.
     * @param noNamespaceSchemaLocation Use this in cases where the schema for your document does not have a namespace (eg.
     *        "http://www.dummyUri.com/myDocumentSchema.xsd"). May be null.
     * @param schemaLocations Space-separated list of namespace/schema pairings (eg. "http://www.dummyUri.com/myDocumentSchema
     *        http://www.dummyUri.com/myDocumentSchema.xsd"). May be null.
     *
     * @return a String that is the signed document
     *
     * @throws IXSILException if IXSIL throws an exception when creating the signature
     * @throws IOException if there is an exception when serializing the signature to a string
     */
    public String xmlFragmentSignData(String toBeSignedXmlDataById, InputStream xmlResourceData, String xpathSignaturePosition,
        String noNamespaceSchemaLocation, String schemaLocations)
    throws IXSILException, IOException, RFC2253NameParserException
    {
        return m_signer.xmlFragmentSignData(toBeSignedXmlDataById, xmlResourceData, xpathSignaturePosition,
            noNamespaceSchemaLocation, schemaLocations);
    }

}
