//===========================================================================
//
// Copyright (c)  2002-2010 Entrust.  All rights reserved.
// 
//===========================================================================

package com.entrust.toolkit.examples.xml.utils;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileNotFoundException;

import com.entrust.toolkit.exceptions.UserFatalException;
import com.entrust.toolkit.exceptions.UserNotLoggedInException;
import com.entrust.toolkit.User;
import com.entrust.toolkit.xencrypt.core.Encryptor;
import com.entrust.toolkit.xencrypt.init.XMLEInit;
import com.entrust.toolkit.xencrypt.exceptions.EncryptionHandlerException;
import com.entrust.toolkit.xencrypt.exceptions.EncryptorException;

import iaik.x509.X509Certificate;
import iaik.ixsil.init.IXSILInit;
import iaik.ixsil.exceptions.InitException;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * ElementEncryptor is a utility class that encrypts all XML elements of a particular name.
 *<P>
 * The Decryption Transform example uses this class to create a sample document that
 * contains a mixture of encrypted and/or signed XML content.
 *</P>
 */
public class ElementEncryptor {

    /**
    * A User is required so the Toolkit can validate encryption certificates.
    */
    private static User m_user = null ;

    /**
    * The Encryptor instance encrypts XML
    */
    private Encryptor m_encryptor = null ;

    /**
    * Encryption certificate for recipients
    */
    private X509Certificate[] m_certs = null ;

    /**
     * The tag name of the elements to encrypt
     */
    private String m_elementToEncrypt = null;

    /**
     * Controls whether the elements tags and attributes are encrypted, or only its content
     */
    private boolean m_contentOnly = true;

    /**
     * Just a counter for the number of encryptions
     */
    private int m_encryptedDataCounter = 0;

    /**
     * Constructor
     */
    public ElementEncryptor(User user,
                            X509Certificate[] recipients,
                            String elementsToEncrypt)
    {
        m_elementToEncrypt = elementsToEncrypt;
        m_user = user ;
        m_certs = recipients ;
        m_encryptedDataCounter = 0;
    }

    /**
    * Encrypts all elements of a particular name.
    */
    public void encrypt(XMLEInit initializer, String xmlPlaintextFile)
    throws EncryptionHandlerException, FileNotFoundException, UserNotLoggedInException
    {
	    // Create an Encryptor instance
	    m_encryptor = new Encryptor(initializer, new FileInputStream(xmlPlaintextFile));

	    // Provide a trust manager, so the encryption certificates are validated
        m_encryptor.setTrustmanager(m_user);

        // Encrypt particular DOM elements for particular recipient(s)
        Node root = m_encryptor.getDocument().getDocumentElement();

        // Encrypt each DOM element with the specified name.
        encryptElements(root);
    }


    /**
     * If set to true, only the element content will be encrypted,
     * not the element's tags and attributes.  The default is 'false'.
     *
     * @param contentOnly The content-only setting
     */
    public void setContentOnly(boolean contentOnly)
    {
        m_contentOnly = contentOnly;
    }

    /**
     * Write the DOM document to a file.
     *
     * @param encryptedFile The name of the file to be written
     */
    public void writeResult(String encryptedFile)
    throws EncryptionHandlerException,
           FileNotFoundException
    {
        m_encryptor.toOutputStream(new FileOutputStream(encryptedFile));
    }

    /**
     * Returns the count of the number of elements encrypted.
     */
    public int getNumberOfElementsEncrypted()
    {
        return m_encryptedDataCounter;
    }

    /**
     * Traverses the DOM tree and encrypts any element with the specified name.
     */
    private void encryptElements(Node thisNode)
    throws EncryptionHandlerException
    {

        NodeList childNodes = thisNode.getChildNodes();

        for (int i = 0; i < childNodes.getLength(); i++) {

            Node child = childNodes.item(i);
            if (child.getNodeType() == Node.ELEMENT_NODE) {

                String nodeName = child.getNodeName();
                if(nodeName.equals(m_elementToEncrypt)) {
                    encryptElement((Element) child, m_contentOnly, m_certs);
                }
            }
            encryptElements(child);
        }
    }


    /**
     * Encrypts one DOM element for one or more recipients.  The contentOnly flag controls
     * whether the whole element is encrypted or only its content.
     */
    private void encryptElement(Element element, boolean contentOnly, X509Certificate[] certs)
    throws EncryptionHandlerException
    {
        // Attach recipients
        for (int i=0; i< certs.length; i++) {
            m_encryptor.setRecipient(element, certs[i]);
        }

        //encrypt one element
        if(contentOnly){
            m_encryptor.encryptContent(element);
            m_encryptor.updateContent(element);
        }
        else{
            m_encryptor.encryptElement(element);
            m_encryptor.updateElement(element);
        }
        

        // just for displaying information
        m_encryptedDataCounter++;
    }
}